<?php
/**
 * Banco do Brasil Gateway
 * 
 * Integration with BB PIX API
 * 
 * @package HNG_Commerce
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class HNG_Gateway_BB extends HNG_Gateway_Base {
    
    public $id = 'bancodobrasil';
    public $title = 'Banco do Brasil';
    public $api_url = 'https://api.bb.com.br/pix/v1';
    
    public function __construct() {
        parent::__construct();
        $this->supports = ['pix'];
        
        add_action('wp_ajax_hng_bb_create_payment', [$this, 'ajax_create_payment']);
        add_action('wp_ajax_nopriv_hng_bb_create_payment', [$this, 'ajax_create_payment']);
    }
    
    public function get_settings() {
        return [
            'enabled' => get_option('hng_bb_enabled', 'no'),
            'developer_key' => get_option('hng_bb_developer_key', ''),
            'client_id' => get_option('hng_bb_client_id', ''),
            'client_secret' => get_option('hng_bb_client_secret', ''),
        ];
    }
    
    public function process_payment($order_id, $payment_data) {
        $order = new HNG_Order($order_id);
        
        try {
            return $this->process_pix($order, $payment_data);
        } catch (Exception $e) {
            return new WP_Error('payment_error', $e->getMessage());
        }
    }
    
    private function process_pix($order, $payment_data) {
        $token = $this->get_access_token();
        
        if (is_wp_error($token)) return $token;
        
        $txid = substr(md5(uniqid($order->get_id(), true)), 0, 35);
        
        $payload = [
            'calendario' => [
                'expiracao' => 1800,
            ],
            'devedor' => [
                'cpf' => preg_replace('/\D/', '', $payment_data['document'] ?? ''),
                'nome' => $payment_data['customer_name'] ?? '',
            ],
            'valor' => [
                'original' => number_format($order->get_total(), 2, '.', ''),
            ],
            'chave' => get_option('hng_bb_pix_key', ''),
            /* translators: %1$s: order ID */
            'solicitacaoPagador' => sprintf(esc_html__('Pedido #%1$s', 'hng-commerce'), $order->get_id()),
        ];
        
        $response = $this->request('PUT', '/cob/' . $txid, $payload, $token);
        
        if (is_wp_error($response)) return $response;
        
        $pix_data = [
            'qr_code' => $response['pixCopiaECola'] ?? '',
            'txid' => $txid,
        ];
        
        update_post_meta($order->get_id(), '_bb_txid', $txid);
        update_post_meta($order->get_id(), '_bb_pix_data', $pix_data);
        
        return [
            'success' => true,
            'payment_method' => 'pix',
            'pix_data' => $pix_data,
            'redirect_url' => home_url('/pagamento/pix?order_id=' . $order->get_id()),
        ];
    }
    
    private function get_access_token() {
        $settings = $this->get_settings();
        
        $url = 'https://oauth.bb.com.br/oauth/token';
        
        $args = [
            'method' => 'POST',
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($settings['client_id'] . ':' . $settings['client_secret']),
                'Content-Type' => 'application/x-www-form-urlencoded',
            ],
            'body' => 'grant_type=client_credentials&scope=cob.write cob.read',
        ];
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) return $response;
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        return $data['access_token'] ?? new WP_Error('no_token', 'Token não obtido');
    }
    
    private function request($method, $endpoint, $data, $token) {
        $settings = $this->get_settings();
        $url = $this->api_url . $endpoint;
        
        $args = [
            'method' => $method,
            'headers' => [
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
                'X-Developer-Application-Key' => $settings['developer_key'],
            ],
            'body' => wp_json_encode($data),
        ];
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) return $response;
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $code = wp_remote_retrieve_response_code($response);
        
        if ($code >= 400) {
            return new WP_Error('api_error', $body['mensagem'] ?? 'Erro na API');
        }
        
        return $body;
    }
    
    public function ajax_create_payment() {
        check_ajax_referer('HNG Commerce', 'nonce');
        $post = function_exists('wp_unslash') ? wp_unslash($_POST) : $_POST;

        $order_id = absint($post['order_id'] ?? 0);
        $payment_data = isset($post['payment_data']) && is_array($post['payment_data']) ? $post['payment_data'] : [];
        
        if (!$order_id) {
            wp_send_json_error(['message' => __('Pedido inválido.', 'hng-commerce')]);
        }
        
        $result = $this->process_payment($order_id, $payment_data);
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        
        wp_send_json_success($result);
    }
}

new HNG_Gateway_BB();

// Handler de renovação manual de assinatura (Banco do Brasil)
add_action('hng_subscription_manual_renewal', function($subscription_id, $order_id, $payment_method) {
    try {
        $subscription = new HNG_Subscription($subscription_id);
        if ($subscription->get_gateway() !== 'bancodobrasil') return;

        $class = 'HNG_Gateway_BB';
        if (!class_exists($class)) {
            $path = HNG_COMMERCE_PATH . 'gateways/bancodobrasil/class-gateway-bb.php';
            if (file_exists($path)) require_once $path;
        }
        if (!class_exists($class)) return;

        $gw = method_exists($class, 'instance') ? $class::instance() : new $class();
        if (!$gw->is_configured()) return;

        $order = new HNG_Order($order_id);
        $customer_email = $order->get_customer_email();
        $amount = $subscription->get_amount();

        $payment_data = [
            'order_id' => $order_id,
            'amount' => $amount,
            'customer_email' => $customer_email,
            'description' => sprintf('Renovação Assinatura #%d', $subscription_id),
        ];

        if ($payment_method === 'pix' && method_exists($gw, 'process_pix')) {
            $result = $gw->process_pix(new HNG_Order($order_id), $payment_data);
        } else {
            $result = $gw->process_payment($order_id, $payment_data);
        }

        if (is_wp_error($result)) {
            if (function_exists('hng_files_log_append')) {
                hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-bb.log', '[BB Renovaá§á¡o] ' . $result->get_error_message() . PHP_EOL);
            }
            return;
        }
        if (is_array($result)) update_post_meta($order_id, '_payment_data', $result);

        $candidates = ['paymentUrl','payment_url','qr_code','txid','redirect_url','boleto_url'];
        $payment_url = '';
        if (is_array($result)) { foreach ($candidates as $k) { if (!empty($result[$k])) { $payment_url = $result[$k]; break; } } }
        if (!empty($payment_url)) update_post_meta($order_id, '_payment_url', $payment_url);

    } catch (Exception $e) {
        if (function_exists('hng_files_log_append')) {
            hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-bb.log', '[BB Renovaá§á¡o] Exception: ' . $e->getMessage() . PHP_EOL);
        }
    }
}, 10, 3);
