<?php
/**
 * C6 Bank Gateway
 * 
 * Integration with C6 Bank PIX API
 * 
 * @package HNG_Commerce
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class HNG_Gateway_C6Bank extends HNG_Gateway_Base {
    
    public $id = 'c6bank';
    public $title = 'C6 Bank';
    public $api_url = 'https://api.c6bank.com.br/v1';
    
    public function __construct() {
        parent::__construct();
        $this->supports = ['pix'];
        
        add_action('wp_ajax_hng_c6_create_payment', [$this, 'ajax_create_payment']);
        add_action('wp_ajax_nopriv_hng_c6_create_payment', [$this, 'ajax_create_payment']);
    }
    
    public function get_settings() {
        return [
            'enabled' => get_option('hng_c6_enabled', 'no'),
            'client_id' => get_option('hng_c6_client_id', ''),
            'client_secret' => get_option('hng_c6_client_secret', ''),
            'pix_key' => get_option('hng_c6_pix_key', ''),
        ];
    }
    
    public function process_payment($order_id, $payment_data) {
        $order = new HNG_Order($order_id);
        
        try {
            return $this->process_pix($order, $payment_data);
        } catch (Exception $e) {
            return new WP_Error('payment_error', $e->getMessage());
        }
    }
    
    private function process_pix($order, $payment_data) {
        $settings = $this->get_settings();
        $token = $this->get_access_token();
        
        if (is_wp_error($token)) return $token;
        
        $payload = [
            'amount' => intval($order->get_total() * 100),
            /* translators: %1$s: order ID */
            'description' => sprintf(esc_html__('Pedido #%1$s', 'hng-commerce'), $order->get_id()),
            'payer' => [
                'name' => $payment_data['customer_name'] ?? '',
                'document' => preg_replace('/\D/', '', $payment_data['document'] ?? ''),
            ],
            'expiration' => 1800,
            'pix_key' => $settings['pix_key'],
        ];
        
        $response = $this->request('POST', '/pix/payments', $payload, $token);
        
        if (is_wp_error($response)) return $response;
        
        $pix_data = [
            'qr_code' => $response['qr_code'] ?? '',
            'qr_code_base64' => $response['qr_code_image'] ?? '',
            'payment_id' => $response['id'] ?? '',
        ];
        
        update_post_meta($order->get_id(), '_c6_payment_id', $response['id'] ?? '');
        update_post_meta($order->get_id(), '_c6_pix_data', $pix_data);
        
        // Calcular e registrar taxas do plugin HNG
        $this->register_hng_fees($order, 'pix', $response['id'] ?? '');
        
        return [
            'success' => true,
            'payment_method' => 'pix',
            'pix_data' => $pix_data,
            'redirect_url' => home_url('/pagamento/pix?order_id=' . $order->get_id()),
        ];
    }
    
    private function get_access_token() {
        $settings = $this->get_settings();
        
        $url = $this->api_url . '/oauth/token';
        
        $args = [
            'method' => 'POST',
            'headers' => ['Content-Type' => 'application/x-www-form-urlencoded'],
            'body' => [
                'client_id' => $settings['client_id'],
                'client_secret' => $settings['client_secret'],
                'grant_type' => 'client_credentials',
            ],
        ];
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) return $response;
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        return $data['access_token'] ?? new WP_Error('no_token', 'Token não obtido');
    }
    
    private function request($method, $endpoint, $data, $token) {
        $url = $this->api_url . $endpoint;
        
        $args = [
            'method' => $method,
            'headers' => [
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($data),
        ];
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) return $response;
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $code = wp_remote_retrieve_response_code($response);
        
        if ($code >= 400) {
            return new WP_Error('api_error', $body['message'] ?? 'Erro na API');
        }
        
        return $body;
    }
    
    public function ajax_create_payment() {
        check_ajax_referer('HNG Commerce', 'nonce');
        $post = function_exists('wp_unslash') ? wp_unslash($_POST) : $_POST;

        $order_id = absint($post['order_id'] ?? 0);
        $payment_data = isset($post['payment_data']) && is_array($post['payment_data']) ? $post['payment_data'] : [];
        
        if (!$order_id) {
            wp_send_json_error(['message' => __('Pedido inválido.', 'hng-commerce')]);
        }
        
        $result = $this->process_payment($order_id, $payment_data);
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        
        wp_send_json_success($result);
    }
    
    /**
     * Calcular e registrar taxas do plugin HNG
     * 
     * @param HNG_Order $order Objeto do pedido
     * @param string $payment_method Método de pagamento (pix)
     * @param string $external_ref Referência externa do pagamento
     */
    private function register_hng_fees($order, $payment_method, $external_ref = '') {
        if (!class_exists('HNG_Fee_Calculator')) {
            return;
        }
        
        $calc = HNG_Fee_Calculator::instance();
        
        // Determinar tipo de produto (físico por padrão)
        $product_type = 'physical';
        
        // Calcular todas as taxas
        $fee_data = $calc->calculate_all_fees(
            $order->get_total(),
            $product_type,
            $this->id,
            $payment_method
        );
        
        // Salvar dados das taxas no pedido
        update_post_meta($order->get_id(), '_hng_fee_data', $fee_data);
        update_post_meta($order->get_id(), '_hng_plugin_fee', $fee_data['plugin_fee']);
        update_post_meta($order->get_id(), '_hng_gateway_fee', $fee_data['gateway_fee']);
        update_post_meta($order->get_id(), '_hng_net_amount', $fee_data['net_amount']);
        update_post_meta($order->get_id(), '_hng_tier', $fee_data['tier']);
        
        // Registrar no Ledger se disponível
        if (class_exists('HNG_Ledger')) {
            HNG_Ledger::add_entry([
                'type' => 'charge',
                'order_id' => $order->get_id(),
                'external_ref' => $external_ref,
                'gross_amount' => $fee_data['gross_amount'],
                'fee_amount' => $fee_data['plugin_fee'] + $fee_data['gateway_fee'],
                'net_amount' => $fee_data['net_amount'],
                'status' => 'pending',
                'meta' => [
                    'gateway' => $this->id,
                    'method' => $payment_method,
                    'plugin_fee' => $fee_data['plugin_fee'],
                    'gateway_fee' => $fee_data['gateway_fee'],
                    'tier' => $fee_data['tier']
                ]
            ]);
        }
        
        // Registrar transação no Fee Calculator
        $calc->register_transaction($order->get_id(), $fee_data);
        
        $this->log('HNG Fees registered', $fee_data);
    }
}

new HNG_Gateway_C6Bank();

// Handler de renovação manual de assinatura (C6 Bank)
add_action('hng_subscription_manual_renewal', function($subscription_id, $order_id, $payment_method) {
    try {
        $subscription = new HNG_Subscription($subscription_id);
        if ($subscription->get_gateway() !== 'c6bank') return;

        $class = 'HNG_Gateway_C6Bank';
        if (!class_exists($class)) {
            $path = HNG_COMMERCE_PATH . 'gateways/c6bank/class-gateway-c6bank.php';
            if (file_exists($path)) require_once $path;
        }
        if (!class_exists($class)) return;

        $gw = method_exists($class, 'instance') ? $class::instance() : new $class();
        if (!$gw->is_configured()) return;

        $order = new HNG_Order($order_id);
        $customer_email = $order->get_customer_email();
        $amount = $subscription->get_amount();

        $payment_data = [
            'order_id' => $order_id,
            'amount' => $amount,
            'customer_email' => $customer_email,
            'description' => sprintf('Renovação Assinatura #%d', $subscription_id),
        ];

        if ($payment_method === 'pix' && method_exists($gw, 'process_pix')) {
            $result = $gw->process_pix(new HNG_Order($order_id), $payment_data);
        } else {
            $result = $gw->process_payment($order_id, $payment_data);
        }

        if (is_wp_error($result)) {
            if (function_exists('hng_files_log_append')) { hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-c6bank.log', '[C6Bank Renovaá§á¡o] ' . $result->get_error_message() . PHP_EOL); }
            return;
        }
        if (is_array($result)) update_post_meta($order_id, '_payment_data', $result);

        $candidates = ['redirect_url','payment_url','paymentUrl','pix_data','qr_code','qr_code_url'];
        $payment_url = '';
        if (is_array($result)) { foreach ($candidates as $k) { if (!empty($result[$k])) { $payment_url = $result[$k]; break; } } }
        if (!empty($payment_url)) update_post_meta($order_id, '_payment_url', $payment_url);

    } catch (Exception $e) {
        if (function_exists('hng_files_log_append')) { hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-c6bank.log', '[C6Bank Renovaá§á¡o] Exception: ' . $e->getMessage() . PHP_EOL); }
    }
}, 10, 3);
