<?php
/**
 * Gateway Banco Inter
 * 
 * @package HNG_Commerce
 * @since 1.0.6
 */

if (!defined('ABSPATH')) {
    exit;
}

// Handler de renovação manual de assinatura (Banco Inter)
add_action('hng_subscription_manual_renewal', function($subscription_id, $order_id, $payment_method) {
    try {
        $subscription = new HNG_Subscription($subscription_id);
        if ($subscription->get_gateway() !== 'inter') return;

        $class = 'HNG_Gateway_Inter';
        if (!class_exists($class)) {
            $path = HNG_COMMERCE_PATH . 'gateways/inter/class-gateway-inter.php';
            if (file_exists($path)) require_once $path;
        }
        if (!class_exists($class)) return;

        $gw = method_exists($class, 'instance') ? $class::instance() : new $class();
        if (!$gw->is_configured()) return;

        $order = new HNG_Order($order_id);
        $customer_email = $order->get_customer_email();
        $amount = $subscription->get_amount();

        $payment_data = [
            'order_id' => $order_id,
            'amount' => $amount,
            'customer_email' => $customer_email,
            'description' => sprintf('Renovação Assinatura #%d', $subscription_id),
        ];

        if ($payment_method === 'pix' && method_exists($gw, 'create_pix_payment')) {
            $result = $gw->create_pix_payment($order_id, $payment_data);
        } else {
            $result = $gw->process_payment($order_id, $payment_data);
        }

        if (is_wp_error($result)) {
            if (function_exists('hng_files_log_append')) {
                hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-inter.log', '[Inter Renovaá§á¡o] ' . $result->get_error_message() . PHP_EOL);
            }
            return;
        }
        if (is_array($result)) update_post_meta($order_id, '_payment_data', $result);

        $candidates = ['encodedImage','payload','qr_code','payment_url','redirect_url'];
        $payment_url = '';
        if (is_array($result)) { foreach ($candidates as $k) { if (!empty($result[$k])) { $payment_url = $result[$k]; break; } } }
        if (!empty($payment_url)) update_post_meta($order_id, '_payment_url', $payment_url);

    } catch (Exception $e) {
        if (function_exists('hng_files_log_append')) {
            hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-inter.log', '[Inter Renovaá§á¡o] Exception: ' . $e->getMessage() . PHP_EOL);
        }
    }
}, 10, 3);


class HNG_Gateway_Inter extends HNG_Payment_Gateway {
    
    public $id = 'inter';
    protected $name = 'Banco Inter';
    public $description = 'Banco Inter - PIX gratuito e APIs abertas';
    
    private $api_url_production = 'https://cdpj.partners.bancointer.com.br';
    private $api_url_sandbox = 'https://cdpj.partners.hml.bancointer.com.br';
    
    /**
     * Construtor
     */
    public function __construct() {
        parent::__construct();
        
        $this->supports = ['pix', 'boleto', 'refunds', 'webhooks'];
        
        // AJAX endpoints
        add_action('wp_ajax_hng_inter_create_payment', [$this, 'ajax_create_payment']);
        add_action('wp_ajax_nopriv_hng_inter_create_payment', [$this, 'ajax_create_payment']);
    }
    
    /**
     * Retorna capabilities do gateway
     */
    public function get_capabilities() {
        return [
            'pix' => [
                'enabled' => true,
                'qr_code' => true,
                'expiration' => true,
                'refund' => true,
                'max_amount' => null,
                'instant' => true,
                'fee' => 0 // PIX GRATUITO!
            ],
            'boleto' => [
                'enabled' => true,
                'barcode' => true,
                'expiration' => true,
                'max_amount' => null
            ],
            'credit_card' => [
                'enabled' => false
            ],
            'split' => false,
            'webhook' => true,
            'refund' => [
                'partial' => true,
                'full' => true
            ]
        ];
    }
    
    /**
     * Criar pagamento PIX
     */
    public function create_pix_payment($order_id, $payment_data) {
        $order = new HNG_Order($order_id);
        $amount = $order->get_total();
        
        // Calcular taxas (PIX Inter é gratuito, mas pode ter taxa do plugin)
        if (!class_exists('HNG_Fee_Calculator')) {
            return new WP_Error('fee_calculator_missing', 'Fee calculator não disponível');
        }
        
        $fee_calc = new HNG_Fee_Calculator();
        $fee_result = $fee_calc->calculate($amount, 'pix', $this->id);
        
        // Montar payload PIX
        $payload = [
            'calendario' => [
                'expiracao' => 3600 // 1 hora
            ],
            'devedor' => [
                'cpf' => preg_replace('/[^0-9]/', '', $order->get_meta('_billing_cpf')),
                'nome' => $order->get_customer_name()
            ],
            'valor' => [
                'original' => number_format($amount, 2, '.', '')
            ],
            'chave' => get_option('hng_inter_pix_key', ''), // Chave PIX da conta
            'solicitacaoPagador' => 'Pedido #' . $order_id,
            'infoAdicionais' => [
                [
                    'nome' => 'Pedido',
                    'valor' => (string) $order_id
                ]
            ]
        ];
        
        $response = $this->api_request('/pix/v2/cob', 'POST', $payload);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $loc = $response['loc'] ?? null;
        if (!$loc) {
            return new WP_Error('pix_creation_failed', 'Location não retornada');
        }
        
        // Buscar QR Code
        $qr_response = $this->api_request('/pix/v2/cob/' . $response['txid'], 'GET');
        
        if (is_wp_error($qr_response)) {
            return $qr_response;
        }
        
        // Salvar no ledger
        if (class_exists('HNG_Ledger')) {
            HNG_Ledger::add_entry([
                'order_id' => $order_id,
                'type' => 'charge',
                'method' => 'pix',
                'status' => 'pending',
                'external_ref' => $response['txid'],
                'gross_amount' => $amount,
                'fee_amount' => $fee_result['total'],
                'net_amount' => $fee_result['net'],
                'gateway' => $this->id,
                'meta' => [
                    'txid' => $response['txid'],
                    'loc_id' => $loc['id'] ?? ''
                ]
            ]);
        }
        
        // Gerar QR Code base64
        $qr_code_text = $qr_response['pixCopiaECola'] ?? '';
        $qr_code_base64 = $this->generate_qr_code_base64($qr_code_text);
        
        return [
            'id' => $response['txid'],
            'encodedImage' => $qr_code_base64,
            'payload' => $qr_code_text,
            'expirationDate' => gmdate('Y-m-d\TH:i:s\Z', time() + 3600),
            'fee_total' => $fee_result['total'],
            'net_amount' => $fee_result['net']
        ];
    }
    
    /**
     * Consultar status do PIX
     */
    public function get_pix_status($charge_id) {
        $response = $this->api_request('/pix/v2/cob/' . $charge_id, 'GET');
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        return [
            'status' => $this->normalize_status($response['status'] ?? ''),
            'raw' => $response
        ];
    }
    
    /**
     * Cancelar PIX (não suportado - expira automaticamente)
     */
    public function cancel_pix($charge_id) {
        return new WP_Error('pix_cancel_not_supported', 'Inter: PIX expira automaticamente');
    }
    
    /**
     * Reembolsar PIX
     */
    public function refund_pix($charge_id, $amount = null) {
        // Inter API não tem endpoint específico de refund PIX
        // Precisa fazer PIX de volta manualmente
        return new WP_Error('refund_manual', 'Reembolso PIX Inter deve ser feito manualmente');
    }
    
    /**
     * Gerar QR Code base64
     */
    private function generate_qr_code_base64($text) {
        // Simples geração via API externa ou biblioteca
        // Por enquanto retornar URL de serviço público
        $encoded_text = urlencode($text);
        return "https://chart.googleapis.com/chart?chs=300x300&cht=qr&chl={$encoded_text}";
    }
    
    /**
     * Normalizar status
     */
    private function normalize_status($status) {
        $map = [
            'ATIVA' => 'PENDING',
            'CONCLUIDA' => 'CONFIRMED',
            'REMOVIDA_PELO_USUARIO_RECEBEDOR' => 'OVERDUE',
            'REMOVIDA_PELO_PSP' => 'OVERDUE'
        ];
        
        return $map[$status] ?? 'PENDING';
    }
    
    /**
     * Requisição à API
     */
    protected function api_request($endpoint, $method = 'GET', $body = null) {
        $client_id = get_option('hng_inter_client_id', '');
        $client_secret = get_option('hng_inter_client_secret', '');
        $cert_path = get_option('hng_inter_cert_path', '');
        $key_path = get_option('hng_inter_key_path', '');
        $environment = get_option('hng_inter_environment', 'sandbox');
        
        if (empty($client_id) || empty($client_secret)) {
            return new WP_Error('inter_not_configured', 'Banco Inter não configurado');
        }
        
        // Obter token OAuth2
        $token = $this->get_access_token();
        if (is_wp_error($token)) {
            return $token;
        }
        
        $base_url = ($environment === 'production') ? $this->api_url_production : $this->api_url_sandbox;
        $url = $base_url . $endpoint;
        
        $args = [
            'method' => $method,
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $token
            ],
            'timeout' => 30
        ];
        
        // Inter exige certificado mTLS
        if (!empty($cert_path) && file_exists($cert_path)) {
            $args['sslcertificate'] = $cert_path;
        }
        if (!empty($key_path) && file_exists($key_path)) {
            $args['sslkey'] = $key_path;
        }
        
        if ($body && in_array($method, ['POST', 'PUT', 'PATCH'])) {
            $args['body'] = wp_json_encode($body);
        }
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $body_response = wp_remote_retrieve_body($response);
        $data = json_decode($body_response, true);
        
        if ($code < 200 || $code >= 300) {
            $error_message = $data['mensagem'] ?? $data['message'] ?? 'Erro desconhecido';
            return new WP_Error('inter_api_error', $error_message, ['code' => $code, 'data' => $data]);
        }
        
        return $data;
    }
    
    /**
     * Obter access token OAuth2
     */
    private function get_access_token() {
        $transient_key = 'hng_inter_access_token';
        $token = get_transient($transient_key);
        
        if ($token) {
            return $token;
        }
        
        // Requisitar novo token
        $client_id = get_option('hng_inter_client_id', '');
        $client_secret = get_option('hng_inter_client_secret', '');
        $environment = get_option('hng_inter_environment', 'sandbox');
        
        $base_url = ($environment === 'production') ? $this->api_url_production : $this->api_url_sandbox;
        
        $response = wp_remote_post($base_url . '/oauth/v2/token', [
            'body' => [
                'client_id' => $client_id,
                'client_secret' => $client_secret,
                'scope' => 'cob.read cob.write pix.read pix.write',
                'grant_type' => 'client_credentials'
            ],
            'timeout' => 30
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!isset($data['access_token'])) {
            return new WP_Error('inter_auth_failed', 'Falha ao obter token');
        }
        
        $token = $data['access_token'];
        $expires_in = $data['expires_in'] ?? 3600;
        
        set_transient($transient_key, $token, $expires_in - 60);
        
        return $token;
    }
    
    /**
     * AJAX: Criar pagamento
     */
    public function ajax_create_payment() {
        check_ajax_referer('hng-checkout', 'nonce');
        
        $post = function_exists('wp_unslash') ? wp_unslash($_POST) : $_POST;
        $order_id = absint($post['order_id'] ?? 0);
        $payment_method = isset($post['payment_method']) ? sanitize_text_field($post['payment_method']) : '';

        if (!$order_id || !$payment_method) {
            wp_send_json_error(['message' => 'Dados inválidos']);
        }

        $raw = is_array($post['payment_data'] ?? null) ? $post['payment_data'] : (is_array($post) ? $post : []);

        $result = null;

        switch ($payment_method) {
            case 'pix':
                $pix_data = [];
                if (is_array($raw)) {
                    $pix_data['customer_name'] = isset($raw['customer_name']) ? sanitize_text_field($raw['customer_name']) : '';
                    $pix_data['document'] = isset($raw['document']) ? preg_replace('/\D/', '', $raw['document']) : '';
                    $pix_data['amount'] = isset($raw['amount']) ? floatval($raw['amount']) : 0;
                }
                $result = $this->create_pix_payment($order_id, $pix_data);
                break;
        }
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        
        wp_send_json_success($result);
    }
}
