<?php
/**
 * Itaú Gateway
 *
 * Integration with Itaú PIX API
 *
 * @package HNG_Commerce
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class HNG_Gateway_Itau extends HNG_Gateway_Base {
    
    public $id = 'itau';
    public $title = 'Itaú';
    public $api_url = 'https://api.itau.com.br/pix_recebimentos/v2';
    
    public function __construct() {
        parent::__construct();
        $this->supports = ['pix'];
        
        add_action('wp_ajax_hng_itau_create_payment', [$this, 'ajax_create_payment']);
        add_action('wp_ajax_nopriv_hng_itau_create_payment', [$this, 'ajax_create_payment']);
    }
    
    public function get_settings() {
        return [
            'enabled' => get_option('hng_itau_enabled', 'no'),
            'client_id' => get_option('hng_itau_client_id', ''),
            'client_secret' => get_option('hng_itau_client_secret', ''),
            'pix_key' => get_option('hng_itau_pix_key', ''),
        ];
    }
    
    public function process_payment($order_id, $payment_data) {
        $order = new HNG_Order($order_id);
        
        try {
            return $this->process_pix($order, $payment_data);
        } catch (Exception $e) {
            return new WP_Error('payment_error', $e->getMessage());
        }
    }
    
    private function process_pix($order, $payment_data) {
        $token = $this->get_access_token();
        
        if (is_wp_error($token)) return $token;
        
        $txid = substr(md5(uniqid($order->get_id(), true)), 0, 35);
        $settings = $this->get_settings();
        
        $payload = [
            'calendario' => [
                'expiracao' => 1800,
            ],
            'devedor' => [
                'cpf' => preg_replace('/\D/', '', $payment_data['document'] ?? ''),
                'nome' => $payment_data['customer_name'] ?? '',
            ],
            'valor' => [
                'original' => number_format($order->get_total(), 2, '.', ''),
            ],
            'chave' => $settings['pix_key'],
            /* translators: %1$s: order ID */
            'solicitacaoPagador' => sprintf(esc_html__('Pedido #%1$s', 'hng-commerce'), $order->get_id()),
            'infoAdicionais' => [
                [
                    'nome' => 'pedido',
                    'valor' => (string) $order->get_id(),
                ]
            ],
        ];
        
        $response = $this->request('PUT', '/cob/' . $txid, $payload, $token);
        
        if (is_wp_error($response)) return $response;
        
        $pix_data = [
            'qr_code' => $response['pixCopiaECola'] ?? '',
            'txid' => $txid,
            'location' => $response['loc']['location'] ?? '',
        ];
        
        update_post_meta($order->get_id(), '_itau_txid', $txid);
        update_post_meta($order->get_id(), '_itau_pix_data', $pix_data);
        
        return [
            'success' => true,
            'payment_method' => 'pix',
            'pix_data' => $pix_data,
            'redirect_url' => home_url('/pagamento/pix?order_id=' . $order->get_id()),
        ];
    }
    
    private function get_access_token() {
        $settings = $this->get_settings();
        
        $url = 'https://sts.itau.com.br/api/oauth/token';
        
        $args = [
            'method' => 'POST',
            'headers' => [
                'Content-Type' => 'application/x-www-form-urlencoded',
            ],
            'body' => [
                'grant_type' => 'client_credentials',
                'client_id' => $settings['client_id'],
                'client_secret' => $settings['client_secret'],
            ],
        ];
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) return $response;
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        return $data['access_token'] ?? new WP_Error('no_token', 'Token não obtido');
    }
    
    private function request($method, $endpoint, $data, $token) {
        $url = $this->api_url . $endpoint;
        
        $args = [
            'method' => $method,
            'headers' => [
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
                'x-itau-correlationID' => wp_generate_uuid4(),
            ],
                'body' => wp_json_encode($data),
        ];
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) return $response;
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $code = wp_remote_retrieve_response_code($response);
        
        if ($code >= 400) {
            return new WP_Error('api_error', $body['mensagem'] ?? 'Erro na API');
        }
        
        return $body;
    }
    
    public function ajax_create_payment() {
        check_ajax_referer('HNG Commerce', 'nonce');
        $post = function_exists('wp_unslash') ? wp_unslash($_POST) : $_POST;

        $order_id = absint($post['order_id'] ?? 0);

        if (!$order_id) {
            wp_send_json_error(['message' => __('Pedido inválido.', 'hng-commerce')]);
        }

        $raw = is_array($post['payment_data'] ?? null) ? $post['payment_data'] : (is_array($post) ? $post : []);
        $payment_data = [
            'payment_method' => sanitize_text_field($raw['payment_method'] ?? ($post['payment_method'] ?? 'pix')),
            'customer_name' => isset($raw['customer_name']) ? sanitize_text_field($raw['customer_name']) : '',
            'document' => isset($raw['document']) ? preg_replace('/\D/', '', $raw['document']) : '',
        ];

        $result = $this->process_payment($order_id, $payment_data);
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        
        wp_send_json_success($result);
    }
}

new HNG_Gateway_Itau();

// Handler de renovação manual de assinatura (Itaú)
add_action('hng_subscription_manual_renewal', function($subscription_id, $order_id, $payment_method) {
    try {
        $subscription = new HNG_Subscription($subscription_id);
        if ($subscription->get_gateway() !== 'itau') return;

        $class = 'HNG_Gateway_Itau';
        if (!class_exists($class)) {
            $path = HNG_COMMERCE_PATH . 'gateways/itau/class-gateway-itau.php';
            if (file_exists($path)) require_once $path;
        }
        if (!class_exists($class)) return;

        $gw = method_exists($class, 'instance') ? $class::instance() : new $class();
        if (!$gw->is_configured()) return;

        $order = new HNG_Order($order_id);
        $customer_email = $order->get_customer_email();
        $amount = $subscription->get_amount();

        $payment_data = [
            'order_id' => $order_id,
            'amount' => $amount,
            'customer_email' => $customer_email,
            'description' => sprintf('Renovação Assinatura #%d', $subscription_id),
        ];

        if ($payment_method === 'pix' && method_exists($gw, 'process_pix')) {
            $result = $gw->process_pix(new HNG_Order($order_id), $payment_data);
        } else {
            $result = $gw->process_payment($order_id, $payment_data);
        }

        if (is_wp_error($result)) {
            if (function_exists('hng_files_log_append')) {
                hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-itau.log', '[Itau Renovaá§á¡o] ' . $result->get_error_message() . PHP_EOL);
            }
            return;
        }
        if (is_array($result)) update_post_meta($order_id, '_payment_data', $result);

        $candidates = ['redirect_url','payment_url','paymentUrl','pix_data','qr_code','txid','loc'];
        $payment_url = '';
        if (is_array($result)) { foreach ($candidates as $k) { if (!empty($result[$k])) { $payment_url = $result[$k]; break; } } }
        if (!empty($payment_url)) update_post_meta($order_id, '_payment_url', $payment_url);

    } catch (Exception $e) {
        if (function_exists('hng_files_log_append')) {
            hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-itau.log', '[Itau Renovaá§á¡o] Exception: ' . $e->getMessage() . PHP_EOL);
        }
    }
}, 10, 3);
