<?php
/**
 * Nubank Gateway (NuPay)
 * 
 * Integration with Nubank NuPay API
 * 
 * @package HNG_Commerce
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class HNG_Gateway_Nubank extends HNG_Gateway_Base {
    
    public $id = 'nubank';
    public $title = 'Nubank (NuPay)';
    public $api_url = 'https://api.nubank.com.br/v1';
    
    public function __construct() {
        parent::__construct();
        $this->supports = ['pix', 'credit_card'];
        
        add_action('wp_ajax_hng_nubank_create_payment', [$this, 'ajax_create_payment']);
        add_action('wp_ajax_nopriv_hng_nubank_create_payment', [$this, 'ajax_create_payment']);
    }
    
    public function get_settings() {
        return [
            'enabled' => get_option('hng_nubank_enabled', 'no'),
            'api_key' => get_option('hng_nubank_api_key', ''),
            'merchant_id' => get_option('hng_nubank_merchant_id', ''),
        ];
    }
    
    public function process_payment($order_id, $payment_data) {
        $order = new HNG_Order($order_id);
        $method = $payment_data['payment_method'] ?? 'pix';
        
        try {
            if ($method === 'pix') {
                return $this->process_pix($order, $payment_data);
            } else {
                return $this->process_credit_card($order, $payment_data);
            }
        } catch (Exception $e) {
            return new WP_Error('payment_error', $e->getMessage());
        }
    }
    
    private function process_pix($order, $payment_data) {
        $settings = $this->get_settings();
        
        $payload = [
            'merchant_id' => $settings['merchant_id'],
            'amount' => intval($order->get_total() * 100),
            /* translators: %s: placeholder */
            'description' => sprintf(esc_html__('Pedido #%s', 'hng-commerce'), $order->get_id()),
            'customer' => [
                'name' => $payment_data['customer_name'] ?? '',
                'email' => $order->get_customer_email(),
                'document' => preg_replace('/\D/', '', $payment_data['document'] ?? ''),
            ],
            'payment_method' => 'pix',
            'expiration_minutes' => 30,
        ];
        
        $response = $this->request('POST', '/payments', $payload);
        
        if (is_wp_error($response)) return $response;
        
        $pix_data = [
            'qr_code' => $response['pix']['qr_code'] ?? '',
            'qr_code_base64' => $response['pix']['qr_code_base64'] ?? '',
            'payment_id' => $response['id'] ?? '',
        ];
        
        update_post_meta($order->get_id(), '_nubank_payment_id', $response['id'] ?? '');
        update_post_meta($order->get_id(), '_nubank_pix_data', $pix_data);
        
        return [
            'success' => true,
            'payment_method' => 'pix',
            'pix_data' => $pix_data,
            'redirect_url' => home_url('/pagamento/pix?order_id=' . $order->get_id()),
        ];
    }
    
    private function process_credit_card($order, $payment_data) {
        $settings = $this->get_settings();
        
        $payload = [
            'merchant_id' => $settings['merchant_id'],
            'amount' => intval($order->get_total() * 100),
            /* translators: %1$s: order ID */
            'description' => sprintf(esc_html__('Pedido #%1$s', 'hng-commerce'), $order->get_id()),
            'customer' => [
                'name' => $payment_data['customer_name'] ?? '',
                'email' => $order->get_customer_email(),
                'document' => preg_replace('/\D/', '', $payment_data['document'] ?? ''),
            ],
            'payment_method' => 'credit_card',
            'card_token' => $payment_data['card_token'] ?? '',
            'installments' => absint($payment_data['installments'] ?? 1),
        ];
        
        $response = $this->request('POST', '/payments', $payload);
        
        if (is_wp_error($response)) return $response;
        
        update_post_meta($order->get_id(), '_nubank_payment_id', $response['id'] ?? '');
        
        if ($response['status'] === 'approved') {
            $order->update_status('processing');
            
            return [
                'success' => true,
                'payment_method' => 'credit_card',
                'status' => 'approved',
                'redirect_url' => $order->get_order_received_url(),
            ];
        }
        
        return [
            'success' => false,
            'status' => $response['status'] ?? 'failed',
            'message' => __('Pagamento não aprovado.', 'hng-commerce'),
        ];
    }
    
    private function request($method, $endpoint, $data) {
        $settings = $this->get_settings();
        $url = $this->api_url . $endpoint;
        
        $args = [
            'method' => $method,
            'headers' => [
                'Authorization' => 'Bearer ' . $settings['api_key'],
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($data),
        ];
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) return $response;
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $code = wp_remote_retrieve_response_code($response);
        
        if ($code >= 400) {
            return new WP_Error('api_error', $body['error'] ?? 'Erro na API');
        }
        
        return $body;
    }
    
    public function ajax_create_payment() {
        check_ajax_referer('HNG Commerce', 'nonce');

        $post = function_exists('wp_unslash') ? wp_unslash($_POST) : $_POST;
        $order_id = absint($post['order_id'] ?? 0);

        if (!$order_id) {
            wp_send_json_error(['message' => __('Pedido inválido.', 'hng-commerce')]);
        }

        $raw = is_array($post['payment_data'] ?? null) ? $post['payment_data'] : (is_array($post) ? $post : []);
        $method = sanitize_text_field($raw['payment_method'] ?? ($post['payment_method'] ?? 'pix'));

        $payment_data = ['payment_method' => $method];

        if ($method === 'pix') {
            $payment_data['customer_name'] = isset($raw['customer_name']) ? sanitize_text_field($raw['customer_name']) : '';
            $payment_data['document'] = isset($raw['document']) ? preg_replace('/\D/', '', $raw['document']) : '';
            $payment_data['amount'] = isset($raw['amount']) ? floatval($raw['amount']) : 0;
        } else {
            $payment_data['card_token'] = isset($raw['card_token']) ? sanitize_text_field($raw['card_token']) : '';
            $payment_data['installments'] = isset($raw['installments']) ? absint($raw['installments']) : 1;
        }

        $result = $this->process_payment($order_id, $payment_data);
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        
        wp_send_json_success($result);
    }
}

new HNG_Gateway_Nubank();

// Handler de renovação manual de assinatura (Nubank)
add_action('hng_subscription_manual_renewal', function($subscription_id, $order_id, $payment_method) {
    try {
        $subscription = new HNG_Subscription($subscription_id);
        if ($subscription->get_gateway() !== 'nubank') return;

        $class = 'HNG_Gateway_Nubank';
        if (!class_exists($class)) {
            $path = HNG_COMMERCE_PATH . 'gateways/nubank/class-gateway-nubank.php';
            if (file_exists($path)) require_once $path;
        }
        if (!class_exists($class)) return;

        $gw = method_exists($class, 'instance') ? $class::instance() : new $class();
        if (!$gw->is_configured()) return;

        $order = new HNG_Order($order_id);
        $customer_email = $order->get_customer_email();
        $amount = $subscription->get_amount();

        $payment_data = [
            'order_id' => $order_id,
            'amount' => $amount,
            'customer_email' => $customer_email,
            'description' => sprintf('Renovação Assinatura #%d', $subscription_id),
        ];

        if ($payment_method === 'pix' && method_exists($gw, 'process_pix')) {
            $result = $gw->process_pix(new HNG_Order($order_id), $payment_data);
        } else {
            $result = $gw->process_payment($order_id, $payment_data);
        }

        if (is_wp_error($result)) {
            if (function_exists('hng_files_log_append')) {
                hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-nubank.log', '[Nubank Renovaá§á¡o] ' . $result->get_error_message() . PHP_EOL);
            }
            return; 
        }
        if (is_array($result)) update_post_meta($order_id, '_payment_data', $result);

        $candidates = ['redirect_url','payment_url','paymentUrl','pix_data','qr_code','qr_code_url'];
        $payment_url = '';
        if (is_array($result)) { foreach ($candidates as $k) { if (!empty($result[$k])) { $payment_url = $result[$k]; break; } } }
        if (!empty($payment_url)) update_post_meta($order_id, '_payment_url', $payment_url);

    } catch (Exception $e) {
        if (function_exists('hng_files_log_append')) {
            hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-nubank.log', '[Nubank Renovaá§á¡o] Exception: ' . $e->getMessage() . PHP_EOL);
        }
    }
}, 10, 3);
