<?php
/**
 * Gateway PicPay
 * 
 * @package HNG_Commerce
 * @since 1.0.6
 */

if (!defined('ABSPATH')) {
    exit;
}

// Handler de renovação manual de assinatura (PicPay)
add_action('hng_subscription_manual_renewal', function($subscription_id, $order_id, $payment_method) {
    try {
        $subscription = new HNG_Subscription($subscription_id);
        if ($subscription->get_gateway() !== 'picpay') return;

        $class = 'HNG_Gateway_PicPay';
        if (!class_exists($class)) {
            $path = HNG_COMMERCE_PATH . 'gateways/picpay/class-gateway-picpay.php';
            if (file_exists($path)) require_once $path;
        }
        if (!class_exists($class)) return;

        $gw = method_exists($class, 'instance') ? $class::instance() : new $class();
        if (!$gw->is_configured()) return;

        $order = new HNG_Order($order_id);
        $customer_email = $order->get_customer_email();
        $amount = $subscription->get_amount();

        $payment_data = [
            'order_id' => $order_id,
            'amount' => $amount,
            'customer_email' => $customer_email,
            'description' => sprintf('Renovação Assinatura #%d', $subscription_id),
        ];

        if ($payment_method === 'pix' && method_exists($gw, 'create_pix_payment')) {
            $result = $gw->create_pix_payment($order_id, $payment_data);
        } else {
            $result = $gw->process_payment($order_id, $payment_data);
        }

        if (is_wp_error($result)) {
            if (function_exists('hng_files_log_append')) {
                hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-picpay.log', '[PicPay Renovaá§á¡o] ' . $result->get_error_message() . PHP_EOL);
            }
            return;
        }
        if (is_array($result)) update_post_meta($order_id, '_payment_data', $result);

        $candidates = ['paymentUrl','payment_url','paymentUrl','redirect_url','qrCode','paymentUrl','paymentUrl','paymentUrl'];
        $payment_url = '';
        if (is_array($result)) { foreach ($candidates as $k) { if (!empty($result[$k])) { $payment_url = $result[$k]; break; } } }
        if (!empty($payment_url)) update_post_meta($order_id, '_payment_url', $payment_url);

    } catch (Exception $e) {
        if (function_exists('hng_files_log_append')) {
            hng_files_log_append(HNG_COMMERCE_PATH . 'logs/gateways-picpay.log', '[PicPay Renovaá§á¡o] Exception: ' . $e->getMessage() . PHP_EOL);
        }
    }
}, 10, 3);


class HNG_Gateway_PicPay extends HNG_Payment_Gateway {
    
    public $id = 'picpay';
    protected $name = 'PicPay';
    public $description = 'PicPay - Carteira Digital Brasileira';
    
    public $api_url = 'https://appws.picpay.com/ecommerce/public';
    
    /**
     * Construtor
     */
    public function __construct() {
        parent::__construct();
        
        $this->supports = ['pix', 'digital_wallet', 'qr_code', 'refunds', 'webhooks'];
        
        // AJAX endpoints
        add_action('wp_ajax_hng_picpay_create_payment', [$this, 'ajax_create_payment']);
        add_action('wp_ajax_nopriv_hng_picpay_create_payment', [$this, 'ajax_create_payment']);
    }
    
    /**
     * Retorna capabilities do gateway
     */
    public function get_capabilities() {
        return [
            'pix' => [
                'enabled' => true,
                'qr_code' => true,
                'expiration' => true,
                'refund' => true,
                'max_amount' => null
            ],
            'digital_wallet' => [
                'enabled' => true,
                'qr_code' => true,
                'deep_link' => true, // Abre app PicPay direto
                'instant_payment' => true
            ],
            'credit_card' => [
                'enabled' => false // PicPay não processa cartão diretamente
            ],
            'boleto' => [
                'enabled' => false
            ],
            'split' => false,
            'webhook' => true,
            'refund' => [
                'partial' => true,
                'full' => true
            ]
        ];
    }
    
    /**
     * Criar pagamento PIX
     */
    public function create_pix_payment($order_id, $payment_data) {
        $order = new HNG_Order($order_id);
        $amount = $order->get_total();
        
        // Calcular taxas
        if (!class_exists('HNG_Fee_Calculator')) {
            return new WP_Error('fee_calculator_missing', 'Fee calculator não disponível');
        }
        
        $fee_calc = new HNG_Fee_Calculator();
        $fee_result = $fee_calc->calculate($amount, 'pix', $this->id);
        
        // PicPay usa o mesmo endpoint para PIX e pagamento via app
        $callback_url = add_query_arg([
            'action' => 'hng_picpay_callback',
            'order_id' => $order_id
        ], home_url('/'));
        
        $payload = [
            'referenceId' => 'ORDER_' . $order_id,
            'callbackUrl' => $callback_url,
            'returnUrl' => $order->get_checkout_order_received_url(),
            'value' => (float) $amount,
            'expiresAt' => gmdate('Y-m-d\TH:i:s\Z', time() + 1800), // 30 minutos
            'buyer' => [
                'firstName' => $this->extract_first_name($order->get_customer_name()),
                'lastName' => $this->extract_last_name($order->get_customer_name()),
                'document' => preg_replace('/[^0-9]/', '', $order->get_meta('_billing_cpf')),
                'email' => $order->get_customer_email(),
                'phone' => preg_replace('/[^0-9]/', '', $order->get_meta('_billing_phone'))
            ],
            'paymentMethod' => 'pix' // ou 'picpay' para carteira
        ];
        
        $response = $this->api_request('/payments', 'POST', $payload);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        if (!isset($response['referenceId']) || !isset($response['qrcode'])) {
            return new WP_Error('picpay_creation_failed', 'Resposta inválida da PicPay');
        }
        
        // Salvar no ledger
        if (class_exists('HNG_Ledger')) {
            HNG_Ledger::add_entry([
                'order_id' => $order_id,
                'type' => 'charge',
                'method' => 'pix',
                'status' => 'pending',
                'external_ref' => $response['referenceId'],
                'gross_amount' => $amount,
                'fee_amount' => $fee_result['total'],
                'net_amount' => $fee_result['net'],
                'gateway' => $this->id,
                'meta' => [
                    'payment_url' => $response['paymentUrl'] ?? '',
                    'qr_code' => $response['qrcode']['content'] ?? ''
                ]
            ]);
        }
        
        return [
            'id' => $response['referenceId'],
            'encodedImage' => $response['qrcode']['base64'] ?? '',
            'payload' => $response['qrcode']['content'] ?? '',
            'paymentUrl' => $response['paymentUrl'] ?? '', // Link para abrir app PicPay
            'expirationDate' => $response['expiresAt'] ?? gmdate('Y-m-d\TH:i:s\Z', time() + 1800),
            'fee_total' => $fee_result['total'],
            'net_amount' => $fee_result['net']
        ];
    }
    
    /**
     * Criar pagamento via carteira digital PicPay
     */
    public function create_wallet_payment($order_id, $payment_data) {
        $order = new HNG_Order($order_id);
        $amount = $order->get_total();
        
        // Calcular taxas
        $fee_calc = new HNG_Fee_Calculator();
        $fee_result = $fee_calc->calculate($amount, 'digital_wallet', $this->id);
        
        $callback_url = add_query_arg([
            'action' => 'hng_picpay_callback',
            'order_id' => $order_id
        ], home_url('/'));
        
        $payload = [
            'referenceId' => 'ORDER_' . $order_id,
            'callbackUrl' => $callback_url,
            'returnUrl' => $order->get_checkout_order_received_url(),
            'value' => (float) $amount,
            'expiresAt' => gmdate('Y-m-d\TH:i:s\Z', time() + 1800),
            'buyer' => [
                'firstName' => $this->extract_first_name($order->get_customer_name()),
                'lastName' => $this->extract_last_name($order->get_customer_name()),
                'document' => preg_replace('/[^0-9]/', '', $order->get_meta('_billing_cpf')),
                'email' => $order->get_customer_email(),
                'phone' => preg_replace('/[^0-9]/', '', $order->get_meta('_billing_phone'))
            ],
            'paymentMethod' => 'picpay'
        ];
        
        $response = $this->api_request('/payments', 'POST', $payload);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        // Salvar no ledger
        if (class_exists('HNG_Ledger')) {
            HNG_Ledger::add_entry([
                'order_id' => $order_id,
                'type' => 'charge',
                'method' => 'digital_wallet',
                'status' => 'pending',
                'external_ref' => $response['referenceId'],
                'gross_amount' => $amount,
                'fee_amount' => $fee_result['total'],
                'net_amount' => $fee_result['net'],
                'gateway' => $this->id
            ]);
        }
        
        return [
            'id' => $response['referenceId'],
            'paymentUrl' => $response['paymentUrl'] ?? '',
            'qrCode' => $response['qrcode']['content'] ?? '',
            'qrCodeBase64' => $response['qrcode']['base64'] ?? '',
            'expiresAt' => $response['expiresAt'],
            'fee_total' => $fee_result['total'],
            'net_amount' => $fee_result['net']
        ];
    }
    
    /**
     * Consultar status do pagamento
     */
    public function get_pix_status($charge_id) {
        $reference_id = str_replace('ORDER_', '', $charge_id);
        $response = $this->api_request('/payments/' . $reference_id . '/status', 'GET');
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        return [
            'status' => $this->normalize_status($response['status'] ?? ''),
            'raw' => $response
        ];
    }
    
    /**
     * Cancelar pagamento
     */
    public function cancel_pix($charge_id) {
        $reference_id = str_replace('ORDER_', '', $charge_id);
        return $this->api_request('/payments/' . $reference_id . '/cancellations', 'POST', [
            'authorizationId' => $charge_id
        ]);
    }
    
    /**
     * Reembolsar
     */
    public function refund_pix($charge_id, $amount = null) {
        $reference_id = str_replace('ORDER_', '', $charge_id);
        
        // Consultar o pagamento original para obter authorizationId
        $status = $this->get_pix_status($charge_id);
        if (is_wp_error($status)) {
            return $status;
        }
        
        $authorization_id = $status['raw']['authorizationId'] ?? null;
        if (!$authorization_id) {
            return new WP_Error('picpay_no_authorization', 'Authorization ID não encontrado');
        }
        
        return $this->api_request('/payments/' . $reference_id . '/refunds', 'POST', [
            'authorizationId' => $authorization_id,
            'value' => $amount // null = reembolso total
        ]);
    }
    
    /**
     * Normalizar status
     */
    private function normalize_status($status) {
        $map = [
            'created' => 'PENDING',
            'expired' => 'OVERDUE',
            'analysis' => 'PENDING',
            'paid' => 'CONFIRMED',
            'completed' => 'CONFIRMED',
            'refunded' => 'REFUNDED',
            'chargeback' => 'REFUNDED'
        ];
        
        return $map[strtolower((string) $status)] ?? 'PENDING';
    }
    
    /**
     * Extrair primeiro nome
     */
    private function extract_first_name($full_name) {
        $parts = explode(' ', trim($full_name));
        return $parts[0] ?? '';
    }
    
    /**
     * Extrair sobrenome
     */
    private function extract_last_name($full_name) {
        $parts = explode(' ', trim($full_name));
        if (count($parts) > 1) {
            array_shift($parts);
            return implode(' ', $parts);
        }
        return '';
    }
    
    /**
     * Requisição à API
     */
    protected function api_request($endpoint, $method = 'GET', $body = null) {
        $picpay_token = get_option('hng_picpay_token', '');
        $seller_token = get_option('hng_picpay_seller_token', '');
        
        if (empty($picpay_token) || empty($seller_token)) {
            return new WP_Error('picpay_not_configured', 'PicPay não configurado');
        }
        
        $url = $this->api_url . $endpoint;
        
        $args = [
            'method' => $method,
            'headers' => [
                'Content-Type' => 'application/json',
                'x-picpay-token' => $picpay_token,
                'x-seller-token' => $seller_token
            ],
            'timeout' => 30
        ];
        
        if ($body && in_array($method, ['POST', 'PUT', 'PATCH'])) {
            $args['body'] = wp_json_encode($body);
        }
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $body_response = wp_remote_retrieve_body($response);
        $data = json_decode($body_response, true);
        
        if ($code < 200 || $code >= 300) {
            $error_message = $data['message'] ?? 'Erro desconhecido';
            return new WP_Error('picpay_api_error', $error_message, ['code' => $code, 'data' => $data]);
        }
        
        return $data;
    }
    
    /**
     * AJAX: Criar pagamento
     */
    public function ajax_create_payment() {
        check_ajax_referer('hng-checkout', 'nonce');
        
        $post = function_exists('wp_unslash') ? wp_unslash( $_POST ) : $_POST;
        $order_id = absint( $post['order_id'] ?? 0 );
        $payment_method = isset($post['payment_method']) ? sanitize_text_field( $post['payment_method'] ) : '';
        
        if (!$order_id || !$payment_method) {
            wp_send_json_error(['message' => 'Dados inválidos']);
        }
        
        $result = null;
        
        switch ($payment_method) {
            case 'pix':
                $pix_data = [];
                if (is_array($post)) {
                    $pix_data['customer_email'] = isset($post['customer_email']) ? sanitize_email($post['customer_email']) : '';
                    $pix_data['description'] = isset($post['description']) ? sanitize_text_field($post['description']) : '';
                    $pix_data['amount'] = isset($post['amount']) ? floatval($post['amount']) : 0;
                }

                $result = $this->create_pix_payment($order_id, $pix_data);
                break;
            case 'picpay':
            case 'digital_wallet':
                $wallet_data = [];
                if (is_array($post)) {
                    $wallet_data['customer_email'] = isset($post['customer_email']) ? sanitize_email($post['customer_email']) : '';
                    $wallet_data['description'] = isset($post['description']) ? sanitize_text_field($post['description']) : '';
                    $wallet_data['amount'] = isset($post['amount']) ? floatval($post['amount']) : 0;
                }

                $result = $this->create_wallet_payment($order_id, $wallet_data);
                break;
        }
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        
        wp_send_json_success($result);
    }
}
