<?php
/**
 * HNG Commerce - Email Customizer Page
 *
 * @package HNG_Commerce
 */

if (!defined('ABSPATH')) {
    exit;
}

class HNG_Email_Customizer_Page {

    /**
     * Constructor
     */
    public function __construct() {
        // Register AJAX handlers
        add_action('wp_ajax_hng_save_email_template', [$this, 'save_email_template']);
        add_action('wp_ajax_hng_get_email_preview', [$this, 'get_email_preview']);
        add_action('wp_ajax_hng_send_test_email', [$this, 'send_test_email']);
        add_action('wp_ajax_hng_reset_email_template', [$this, 'reset_email_template']);
    }

    /**
     * Render the page
     */
    public static function render() {
        // Get current email type from query parameter
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only GET parameter
        $current_type = isset($_GET['email_type']) ? sanitize_text_field(wp_unslash($_GET['email_type'])) : 'new_order';
        
        // Available email types
        $email_types = [
            'new_order' => ['name' => __('Novo Pedido', 'hng-commerce')],
            'order_paid' => ['name' => __('Pedido Pago', 'hng-commerce')],
            'order_cancelled' => ['name' => __('Pedido Cancelado', 'hng-commerce')],
            'new_subscription' => ['name' => __('Nova Assinatura', 'hng-commerce')],
            'subscription_renewed' => ['name' => __('Assinatura Renovada', 'hng-commerce')],
            'subscription_cancelled' => ['name' => __('Assinatura Cancelada', 'hng-commerce')],
            'payment_received' => ['name' => __('Pagamento Recebido', 'hng-commerce')],
            'payment_failed' => ['name' => __('Pagamento Falhou', 'hng-commerce')],
        ];

        // Validate current type
        if (!isset($email_types[$current_type])) {
            $current_type = 'new_order';
        }

        // Define available variables for each email type
        $email_variables = [
            'new_order' => [
                '{{customer_name}}' => __('Nome do cliente', 'hng-commerce'),
                '{{order_id}}' => __('ID do Pedido', 'hng-commerce'),
                '{{order_date}}' => __('Data do Pedido', 'hng-commerce'),
                '{{order_total}}' => __('Total do Pedido', 'hng-commerce'),
                '{{products}}' => __('Lista de Produtos', 'hng-commerce'),
                '{{payment_method}}' => __('Método de Pagamento', 'hng-commerce'),
                '{{tracking_url}}' => __('URL de Rastreamento', 'hng-commerce'),
            ],
            'order_paid' => [
                '{{customer_name}}' => __('Nome do cliente', 'hng-commerce'),
                '{{order_id}}' => __('ID do Pedido', 'hng-commerce'),
                '{{amount_paid}}' => __('Valor Pago', 'hng-commerce'),
                '{{payment_date}}' => __('Data do Pagamento', 'hng-commerce'),
                '{{receipt_url}}' => __('URL do Recibo', 'hng-commerce'),
            ],
            'order_cancelled' => [
                '{{customer_name}}' => __('Nome do cliente', 'hng-commerce'),
                '{{order_id}}' => __('ID do Pedido', 'hng-commerce'),
                '{{cancellation_reason}}' => __('Motivo do Cancelamento', 'hng-commerce'),
                '{{refund_amount}}' => __('Valor do Reembolso', 'hng-commerce'),
            ],
            'new_subscription' => [
                '{{customer_name}}' => __('Nome do cliente', 'hng-commerce'),
                '{{subscription_id}}' => __('ID da Assinatura', 'hng-commerce'),
                '{{plan_name}}' => __('Nome do Plano', 'hng-commerce'),
                '{{renewal_date}}' => __('Data de Renovação', 'hng-commerce'),
            ],
            'subscription_renewed' => [
                '{{customer_name}}' => __('Nome do cliente', 'hng-commerce'),
                '{{subscription_id}}' => __('ID da Assinatura', 'hng-commerce'),
                '{{renewal_amount}}' => __('Valor da Renovação', 'hng-commerce'),
                '{{next_renewal}}' => __('Próxima Renovação', 'hng-commerce'),
            ],
            'subscription_cancelled' => [
                '{{customer_name}}' => __('Nome do cliente', 'hng-commerce'),
                '{{subscription_id}}' => __('ID da Assinatura', 'hng-commerce'),
                '{{cancellation_date}}' => __('Data do Cancelamento', 'hng-commerce'),
            ],
            'payment_received' => [
                '{{customer_name}}' => __('Nome do cliente', 'hng-commerce'),
                '{{amount}}' => __('Valor Recebido', 'hng-commerce'),
                '{{payment_date}}' => __('Data do Pagamento', 'hng-commerce'),
                '{{payment_id}}' => __('ID do Pagamento', 'hng-commerce'),
            ],
            'payment_failed' => [
                '{{customer_name}}' => __('Nome do cliente', 'hng-commerce'),
                '{{order_id}}' => __('ID do Pedido', 'hng-commerce'),
                '{{failure_reason}}' => __('Motivo da Falha', 'hng-commerce'),
                '{{retry_url}}' => __('URL para Tentar Novamente', 'hng-commerce'),
            ],
        ];

        // Get email info for current type
        $email_info = [
            'variables' => isset($email_variables[$current_type]) ? $email_variables[$current_type] : []
        ];

        // Get saved template
        $template = get_option("hng_email_template_{$current_type}", [
            'subject' => '',
            'from_name' => get_bloginfo('name'),
            'from_email' => get_option('admin_email'),
            'header' => '',
            'body' => '',
            'content' => '',
            'footer' => '',
            'styles' => [],
            'logo' => '',
            'header_color' => '#0073aa',
            'button_color' => '#0073aa',
            'text_color' => '#333333',
            'bg_color' => '#ffffff',
            'custom_css' => ''
        ]);

        // Prepare settings for template (alias for template)
        $settings = array_merge([
            'subject' => '',
            'from_name' => get_bloginfo('name'),
            'from_email' => get_option('admin_email'),
            'header' => '',
            'body' => '',
            'content' => '',
            'footer' => '',
            'styles' => [],
            'logo' => '',
            'header_color' => '#0073aa',
            'button_color' => '#0073aa',
            'text_color' => '#333333',
            'bg_color' => '#ffffff',
            'custom_css' => ''
        ], $template);
        
        // Se não tem conteúdo, usar template padrão
        if (empty($settings['content'])) {
            $settings['content'] = self::get_default_content_for_type($current_type);
        }

        // Get recent orders for preview selector
        $orders = [];
        // TODO: Fetch real orders if you have an Order class

        // Enqueue assets
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        wp_enqueue_media();
        
        // Enqueue inline styles
        wp_enqueue_style('wp-editor');
        wp_enqueue_script('jquery');

        // Include template file
        $template_file = HNG_COMMERCE_PATH . 'includes/admin/pages/email-customizer-template.php';
        if (file_exists($template_file)) {
            include $template_file;
        }
        
        // Include styles inline
        $styles_file = HNG_COMMERCE_PATH . 'includes/admin/pages/email-customizer-styles.php';
        if (file_exists($styles_file)) {
            include $styles_file;
        }
        
        // Define global variables for inline script
        echo '<script>
            var hngEmailCustomizer = {
                ajax_url: "' . esc_url(admin_url('admin-ajax.php')) . '",
                nonce: "' . esc_attr(wp_create_nonce('hng_email_customizer')) . '",
                current_type: "' . esc_attr($current_type) . '",
                i18n: {
                    saving: "' . esc_attr__('Salvando...', 'hng-commerce') . '",
                    saved: "' . esc_attr__('Salvo!', 'hng-commerce') . '",
                    error: "' . esc_attr__('Erro ao salvar', 'hng-commerce') . '",
                    sending: "' . esc_attr__('Enviando...', 'hng-commerce') . '",
                    sent: "' . esc_attr__('Email enviado!', 'hng-commerce') . '",
                    send_error: "' . esc_attr__('Erro ao enviar email', 'hng-commerce') . '"
                }
            };
            var ajaxurl = "' . esc_url(admin_url('admin-ajax.php')) . '";
        </script>';
        
        // Include scripts inline
        $scripts_file = HNG_COMMERCE_PATH . 'includes/admin/pages/email-customizer-scripts.php';
        if (file_exists($scripts_file)) {
            include $scripts_file;
        }
    }

    /**
     * AJAX: Save email template
     */
    public function save_email_template() {
        check_ajax_referer('hng_email_customizer', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Sem permissão', 'hng-commerce')]);
        }

        // Get and sanitize POST data
        $post = function_exists('wp_unslash') ? wp_unslash($_POST) : $_POST;
        $email_type = isset($post['email_type']) ? sanitize_text_field($post['email_type']) : '';
        
        if (empty($email_type)) {
            wp_send_json_error(['message' => __('Tipo de email inválido', 'hng-commerce')]);
        }

        $template = [
            'subject' => isset($post['subject']) ? sanitize_text_field($post['subject']) : '',
            'from_name' => isset($post['from_name']) ? sanitize_text_field($post['from_name']) : get_bloginfo('name'),
            'from_email' => isset($post['from_email']) ? sanitize_email($post['from_email']) : get_option('admin_email'),
            'content' => isset($post['email_content']) ? wp_kses_post($post['email_content']) : '',
            'logo' => isset($post['logo']) ? esc_url_raw($post['logo']) : '',
            'header_color' => isset($post['header_color']) ? sanitize_hex_color($post['header_color']) : '#0073aa',
            'button_color' => isset($post['button_color']) ? sanitize_hex_color($post['button_color']) : '#0073aa',
            'text_color' => isset($post['text_color']) ? sanitize_hex_color($post['text_color']) : '#333333',
            'bg_color' => isset($post['bg_color']) ? sanitize_hex_color($post['bg_color']) : '#ffffff',
        ];

        update_option("hng_email_template_{$email_type}", $template);

        wp_send_json_success(['message' => __('Template salvo com sucesso!', 'hng-commerce')]);
    }

    /**
     * AJAX: Get email preview
     */
    public function get_email_preview() {
        check_ajax_referer('hng_email_customizer', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Sem permissão', 'hng-commerce')]);
        }

        // Get POST data
        $post = function_exists('wp_unslash') ? wp_unslash($_POST) : $_POST;
        $email_type = isset($post['email_type']) ? sanitize_text_field($post['email_type']) : '';
        
        if (empty($email_type)) {
            wp_send_json_error(['message' => __('Tipo de email inválido', 'hng-commerce')]);
        }

        // Verificar se há dados "live" (preview em tempo real)
        $has_live_data = isset($post['live_content']);
        
        if ($has_live_data) {
            // Usar dados do formulário para preview em tempo real
            $template = [
                'content' => isset($post['live_content']) ? wp_kses_post($post['live_content']) : '',
                'logo' => isset($post['live_logo']) ? esc_url_raw($post['live_logo']) : '',
                'header_color' => isset($post['live_header_color']) && !empty($post['live_header_color']) ? sanitize_hex_color($post['live_header_color']) : '#0073aa',
                'button_color' => isset($post['live_button_color']) && !empty($post['live_button_color']) ? sanitize_hex_color($post['live_button_color']) : '#0073aa',
                'text_color' => isset($post['live_text_color']) && !empty($post['live_text_color']) ? sanitize_hex_color($post['live_text_color']) : '#333333',
                'bg_color' => isset($post['live_bg_color']) && !empty($post['live_bg_color']) ? sanitize_hex_color($post['live_bg_color']) : '#ffffff',
            ];
        } else {
            // Get saved template
            $template = get_option("hng_email_template_{$email_type}", []);
            
            // Merge with defaults
            $template = array_merge([
                'content' => '',
                'logo' => '',
                'header_color' => '#0073aa',
                'button_color' => '#0073aa',
                'text_color' => '#333333',
                'bg_color' => '#ffffff',
            ], $template);
        }
        
        // Se não tem conteúdo, usar template padrão
        if (empty($template['content'])) {
            $template['content'] = $this->get_default_template_content($email_type);
        }
        
        // Substituir variáveis por dados de exemplo
        $content = $this->replace_variables_with_sample_data($template['content'], $email_type);

        // Build email HTML with proper styling
        $html = $this->build_email_html($template, $content);

        wp_send_json_success(['html' => $html]);
    }
    
    /**
     * Obter conteúdo padrão do template por tipo (estático para uso no render)
     */
    public static function get_default_content_for_type($email_type) {
        $defaults = [
            'new_order' => '<h2>Olá {{customer_name}}!</h2>
<p>Recebemos seu pedido <strong>#{{order_id}}</strong> em {{order_date}}.</p>
<p><strong>Itens do Pedido:</strong></p>
{{products}}
<p><strong>Total:</strong> {{order_total}}</p>
<p><strong>Forma de Pagamento:</strong> {{payment_method}}</p>
<p>Você receberá uma atualização quando seu pedido for enviado.</p>
<p>Obrigado por comprar conosco!</p>',
            
            'order_paid' => '<h2>Pagamento Confirmado!</h2>
<p>Olá {{customer_name}},</p>
<p>Confirmamos o pagamento do seu pedido <strong>#{{order_id}}</strong>.</p>
<p><strong>Valor Pago:</strong> {{amount_paid}}</p>
<p><strong>Data:</strong> {{payment_date}}</p>
<p>Agora estamos preparando seu pedido para envio.</p>',

            'order_cancelled' => '<h2>Pedido Cancelado</h2>
<p>Olá {{customer_name}},</p>
<p>Infelizmente seu pedido <strong>#{{order_id}}</strong> foi cancelado.</p>
<p><strong>Motivo:</strong> {{cancellation_reason}}</p>
<p>Se houver reembolso, o valor de {{refund_amount}} será processado em breve.</p>',

            'new_subscription' => '<h2>Bem-vindo à sua Assinatura!</h2>
<p>Olá {{customer_name}},</p>
<p>Sua assinatura <strong>#{{subscription_id}}</strong> foi ativada com sucesso!</p>
<p><strong>Plano:</strong> {{plan_name}}</p>
<p><strong>Próxima renovação:</strong> {{renewal_date}}</p>',

            'subscription_renewed' => '<h2>Assinatura Renovada!</h2>
<p>Olá {{customer_name}},</p>
<p>Sua assinatura <strong>#{{subscription_id}}</strong> foi renovada automaticamente.</p>
<p><strong>Valor:</strong> {{renewal_amount}}</p>
<p><strong>Próxima renovação:</strong> {{next_renewal}}</p>',

            'subscription_cancelled' => '<h2>Assinatura Cancelada</h2>
<p>Olá {{customer_name}},</p>
<p>Sua assinatura <strong>#{{subscription_id}}</strong> foi cancelada em {{cancellation_date}}.</p>
<p>Esperamos vê-lo novamente em breve!</p>',

            'payment_received' => '<h2>Pagamento Recebido!</h2>
<p>Olá {{customer_name}},</p>
<p>Recebemos seu pagamento de <strong>{{amount}}</strong> em {{payment_date}}.</p>
<p><strong>ID do Pagamento:</strong> {{payment_id}}</p>',

            'payment_failed' => '<h2>Problema no Pagamento</h2>
<p>Olá {{customer_name}},</p>
<p>Não conseguimos processar o pagamento do pedido <strong>#{{order_id}}</strong>.</p>
<p><strong>Motivo:</strong> {{failure_reason}}</p>
<p><a href="{{retry_url}}" style="display:inline-block;background:#0073aa;color:white;padding:12px 24px;text-decoration:none;border-radius:4px;">Tentar Novamente</a></p>',
        ];
        
        return $defaults[$email_type] ?? '<p>Escreva o conteúdo do seu email aqui...</p>';
    }

    /**
     * Obter conteúdo padrão do template por tipo
     */
    private function get_default_template_content($email_type) {
        return self::get_default_content_for_type($email_type);
    }
    
    /**
     * Substituir variáveis por dados de exemplo
     */
    private function replace_variables_with_sample_data($content, $email_type) {
        $sample_data = [
            '{{customer_name}}' => 'João Silva',
            '{{order_id}}' => '12345',
            '{{order_date}}' => date_i18n('d/m/Y H:i'),
            '{{order_total}}' => 'R$ 299,90',
            '{{products}}' => '<ul><li>Produto Exemplo x1 - R$ 149,90</li><li>Produto Teste x1 - R$ 150,00</li></ul>',
            '{{payment_method}}' => 'PIX',
            '{{tracking_url}}' => '#',
            '{{amount_paid}}' => 'R$ 299,90',
            '{{payment_date}}' => date_i18n('d/m/Y'),
            '{{receipt_url}}' => '#',
            '{{cancellation_reason}}' => 'Solicitação do cliente',
            '{{refund_amount}}' => 'R$ 299,90',
            '{{subscription_id}}' => 'SUB-001',
            '{{plan_name}}' => 'Plano Premium',
            '{{renewal_date}}' => date_i18n('d/m/Y', strtotime('+30 days')),
            '{{renewal_amount}}' => 'R$ 49,90',
            '{{next_renewal}}' => date_i18n('d/m/Y', strtotime('+30 days')),
            '{{cancellation_date}}' => date_i18n('d/m/Y'),
            '{{amount}}' => 'R$ 299,90',
            '{{payment_id}}' => 'PAY-78901',
            '{{failure_reason}}' => 'Cartão recusado',
            '{{retry_url}}' => '#',
        ];
        
        return str_replace(array_keys($sample_data), array_values($sample_data), $content);
    }
    
    /**
     * Construir HTML do email
     */
    private function build_email_html($template, $content) {
        $header_color = $template['header_color'] ?: '#0073aa';
        $button_color = $template['button_color'] ?: '#0073aa';
        $text_color = $template['text_color'] ?: '#333333';
        $bg_color = $template['bg_color'] ?: '#ffffff';
        $logo = $template['logo'] ?: '';
        
        $html = '<!DOCTYPE html><html><head><meta charset="UTF-8"><style>';
        $html .= 'body { font-family: Arial, sans-serif; line-height: 1.6; color: ' . esc_attr($text_color) . '; background-color: #f5f5f5; margin: 0; padding: 0; }';
        $html .= '.email-container { max-width: 600px; margin: 0 auto; background-color: ' . esc_attr($bg_color) . '; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }';
        $html .= '.email-header { background-color: ' . esc_attr($header_color) . '; color: white; padding: 30px; text-align: center; }';
        $html .= '.email-logo { max-width: 200px; height: auto; margin-bottom: 10px; }';
        $html .= '.email-body { padding: 30px; }';
        $html .= '.email-body h2 { color: ' . esc_attr($header_color) . '; margin-top: 0; }';
        $html .= '.email-button, a[style*="background"] { background-color: ' . esc_attr($button_color) . ' !important; color: white !important; padding: 12px 24px; text-decoration: none; border-radius: 4px; display: inline-block; }';
        $html .= '.email-footer { background-color: #f9f9f9; padding: 20px; text-align: center; font-size: 12px; color: #666; border-top: 1px solid #eee; }';
        $html .= 'ul { padding-left: 20px; }';
        $html .= '</style></head><body>';
        
        $html .= '<div class="email-container">';
        
        // Header com logo
        $html .= '<div class="email-header">';
        if (!empty($logo)) {
            $html .= '<img src="' . esc_url($logo) . '" class="email-logo" alt="Logo">';
        } else {
            $html .= '<h1 style="margin:0;font-size:24px;">' . esc_html(get_bloginfo('name')) . '</h1>';
        }
        $html .= '</div>';
        
        // Body com conteúdo
        $html .= '<div class="email-body">' . wp_kses_post($content) . '</div>';
        
        // Footer
        $html .= '<div class="email-footer">';
        $html .= '<p>&copy; ' . gmdate('Y') . ' ' . esc_html(get_bloginfo('name')) . '. Todos os direitos reservados.</p>';
        $html .= '</div>';
        
        $html .= '</div>';
        $html .= '</body></html>';
        
        return $html;
    }

    /**
     * AJAX: Send test email
     */
    public function send_test_email() {
        check_ajax_referer('hng_email_customizer', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Sem permissão', 'hng-commerce')]);
        }

        // Get POST data
        $post = function_exists('wp_unslash') ? wp_unslash($_POST) : $_POST;
        $email_type = isset($post['email_type']) ? sanitize_text_field($post['email_type']) : '';
        $test_email = isset($post['test_email']) ? sanitize_email($post['test_email']) : '';
        
        if (empty($email_type) || empty($test_email) || !is_email($test_email)) {
            wp_send_json_error(['message' => __('Email inválido', 'hng-commerce')]);
        }

        // Get template
        $template = get_option("hng_email_template_{$email_type}", []);
        $subject = isset($template['subject']) ? $template['subject'] : __('Email de Teste', 'hng-commerce');
        $header = isset($template['header']) ? $template['header'] : '';
        $body = isset($template['body']) ? $template['body'] : '';
        $footer = isset($template['footer']) ? $template['footer'] : '';

        // Build email content
        $content = '<html><body>';
        $content .= '<div style="max-width: 600px; margin: 0 auto; font-family: Arial, sans-serif;">';
        $content .= wp_kses_post($header);
        $content .= wp_kses_post($body);
        $content .= wp_kses_post($footer);
        $content .= '</div>';
        $content .= '</body></html>';

        // Send email
        $headers = ['Content-Type: text/html; charset=UTF-8'];
        $sent = wp_mail($test_email, $subject, $content, $headers);

        if ($sent) {
            wp_send_json_success(['message' => __('Email de teste enviado com sucesso!', 'hng-commerce')]);
        } else {
            wp_send_json_error(['message' => __('Erro ao enviar email de teste', 'hng-commerce')]);
        }
    }

    /**
     * AJAX: Reset email template to default
     */
    public function reset_email_template() {
        check_ajax_referer('hng_email_customizer', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Sem permissão', 'hng-commerce')]);
        }

        // Get POST data
        $post = function_exists('wp_unslash') ? wp_unslash($_POST) : $_POST;
        $email_type = isset($post['email_type']) ? sanitize_text_field($post['email_type']) : '';
        
        if (empty($email_type)) {
            wp_send_json_error(['message' => __('Tipo de email inválido', 'hng-commerce')]);
        }

        // Delete custom template to revert to default
        delete_option("hng_email_template_{$email_type}");

        wp_send_json_success(['message' => __('Template restaurado ao padrão!', 'hng-commerce')]);
    }
}

// Initialize
new HNG_Email_Customizer_Page();
