<?php

/**

 * Gateway Management Page

 * Displays and manages all payment gateways

 */



if (!defined('ABSPATH')) {

    exit;

}



class HNG_Gateway_Management_Page {

    

    /**

     * Construtor - registra AJAX handlers

     */

    public function __construct() {

        error_log('HNG Gateway Management: Constructor called');

    }



    /**

     * Método estático para registrar os hooks AJAX

     * Chamado globalmente no carregamento do plugin

     */

    public static function register_ajax_hooks() {

        error_log('HNG Gateway Management: Registering AJAX actions...');

        

        $instance = new self();

        

        add_action('wp_ajax_hng_save_gateway_config', [$instance, 'save_gateway_config']);

        add_action('wp_ajax_nopriv_hng_save_gateway_config', [$instance, 'save_gateway_config']);

        

        add_action('wp_ajax_hng_test_gateway_connection', [$instance, 'test_gateway_connection']);

        add_action('wp_ajax_nopriv_hng_test_gateway_connection', [$instance, 'test_gateway_connection']); // Permitir sem login para debug

        

        add_action('wp_ajax_hng_check_gateway_status', [$instance, 'check_gateway_status']); // Auto-check sem nonce

        add_action('wp_ajax_nopriv_hng_check_gateway_status', [$instance, 'check_gateway_status']); // Permitir sem login

        

        add_action('wp_ajax_hng_quick_test_gateway', [$instance, 'quick_test_gateway']);

        add_action('wp_ajax_nopriv_hng_quick_test_gateway', [$instance, 'quick_test_gateway']);

        

        add_action('wp_ajax_hng_toggle_gateway', [$instance, 'toggle_gateway']);

        add_action('wp_ajax_nopriv_hng_toggle_gateway', [$instance, 'toggle_gateway']);

        

        add_action('wp_ajax_hng_toggle_advanced_integration', [$instance, 'toggle_advanced_integration']);

        add_action('wp_ajax_nopriv_hng_toggle_advanced_integration', [$instance, 'toggle_advanced_integration']);

        

        add_action('wp_ajax_hng_test_all_gateways', [$instance, 'test_all_gateways']);

        add_action('wp_ajax_nopriv_hng_test_all_gateways', [$instance, 'test_all_gateways']);

        

        error_log('HNG Gateway Management: AJAX actions registered successfully');

    }



    /**

     * Render management UI for gateways

     */

    public function render() {

        $nonce = wp_create_nonce('hng-commerce-admin');

        echo '<div class="hng-wrap hng-gateways-page">';

        

        // Header

        echo '<div class="hng-page-header">';

        echo '<div class="hng-header-title">';

        echo '<h1><span class="dashicons dashicons-credit"></span> Gateways de Pagamento</h1>';

        echo '<p class="description">Configure e gerencie seus gateways de pagamento. Apenas um gateway pode estar ativo por vez.</p>';

        echo '</div>';

        echo '<div class="hng-header-actions">';

        echo '<button type="button" class="button button-primary" id="hng-test-all-gateways" title="Testar conectividade de todos os gateways">';

        echo '<span class="dashicons dashicons-yes-alt"></span> Testar Todos';

        echo '</button>';

        echo '</div>';

        echo '</div>';



        // Card informativo de taxas HNG Commerce

        $fee_calculator = class_exists('HNG_Fee_Calculator') ? HNG_Fee_Calculator::instance() : null;

        $current_tier = $fee_calculator ? $fee_calculator->get_current_tier() : 1;

        $tier_data = $fee_calculator ? $fee_calculator->get_tier_data($current_tier) : null;

        $next_tier_info = $fee_calculator ? $fee_calculator->get_next_tier_info() : null;

        $current_gmv = $fee_calculator ? $fee_calculator->get_current_month_gmv() : 0;

        

        echo '<div class="hng-fees-info-card">';

        echo '<div class="fees-card-header">';

        echo '<div class="fees-card-title">';

        echo '<span class="dashicons dashicons-info-outline"></span>';

        echo '<strong>Sobre as Taxas por Transação</strong>';

        echo '</div>';

        echo '<span class="fees-card-toggle" title="Expandir/Recolher"><span class="dashicons dashicons-arrow-down-alt2"></span></span>';

        echo '</div>';

        

        echo '<div class="fees-card-content">';

        

        // Descrição geral

        echo '<div class="fees-description">';

        echo '<p>Cada venda processada pelo HNG Commerce inclui <strong>duas taxas</strong>:</p>';

        echo '<ul>';

        echo '<li><strong>Taxa do Gateway:</strong> Cobrada pelo processador de pagamentos (varia por gateway e método)</li>';

        echo '<li><strong>Taxa do Plugin HNG:</strong> Taxa escalonada baseada no seu GMV mensal (volume de vendas)</li>';

        echo '</ul>';

        echo '</div>';

        

        // Tier atual

        if ($tier_data) {

            echo '<div class="fees-tier-info">';

            echo '<div class="tier-badge" style="background-color: ' . esc_attr($tier_data['color']) . ';">';

            echo '<span class="tier-icon">' . esc_html($tier_data['icon'] ?? '🎯') . '</span>';

            echo '<span class="tier-name">' . esc_html($tier_data['name']) . '</span>';

            echo '</div>';

            

            echo '<div class="tier-details">';

            echo '<p class="tier-gmv"><strong>Seu GMV este mês:</strong> R$ ' . number_format($current_gmv, 2, ',', '.') . '</p>';

            echo '<div class="tier-fees-list">';

            echo '<span class="fee-item"><strong>Físico:</strong> ' . esc_html($tier_data['fees']['physical']) . '%</span>';

            echo '<span class="fee-item"><strong>Digital:</strong> ' . esc_html($tier_data['fees']['digital']) . '%</span>';

            echo '<span class="fee-item"><strong>Assinatura:</strong> ' . esc_html($tier_data['fees']['subscription']) . '%</span>';

            echo '<span class="fee-item"><strong>Orçamento:</strong> ' . esc_html($tier_data['fees']['quote'] ?? '-') . '%</span>';

            echo '<span class="fee-item"><strong>Agendamento:</strong> ' . esc_html($tier_data['fees']['appointment'] ?? '-') . '%</span>';

            echo '</div>';

            echo '<p class="tier-minimum"><small>* Taxa mínima de R$ 0,50 por transação</small></p>';

            echo '</div>';

            echo '</div>';

        }

        

        // Próximo tier (se não estiver no máximo)

        if ($next_tier_info) {

            echo '<div class="fees-next-tier">';

            echo '<p class="next-tier-msg">';

            echo '🚀 Faltam <strong>R$ ' . number_format($next_tier_info['remaining'], 2, ',', '.') . '</strong> para o tier <strong>' . esc_html($next_tier_info['next_tier_name']) . '</strong>';

            echo '</p>';

            echo '<div class="next-tier-progress">';

            echo '<div class="progress-bar" style="width: ' . esc_attr($next_tier_info['progress_percentage']) . '%;"></div>';

            echo '</div>';

            echo '</div>';

        }

        

        // Tabela completa de tiers

        echo '<div class="fees-tiers-table">';

        echo '<h4>Tabela de Taxas por Tier (HNG Commerce)</h4>';

        echo '<table>';

        echo '<thead><tr><th>Tier</th><th>GMV Mensal</th><th>Físico</th><th>Digital</th><th>Assinatura</th><th>Orçamento</th><th>Agendamento</th></tr></thead>';

        echo '<tbody>';

        if ($fee_calculator) {

            foreach ($fee_calculator->get_all_tiers() as $num => $tier) {

                $is_current = ($num === $current_tier);

                $row_class = $is_current ? 'current-tier' : '';

                echo '<tr class="' . esc_attr($row_class) . '">';

                echo '<td><span class="tier-name-badge" style="background-color: ' . esc_attr($tier['color']) . ';">' . esc_html($tier['icon'] ?? '') . ' ' . esc_html($tier['name']) . '</span></td>';

                if ($tier['gmv_max'] >= PHP_INT_MAX) {

                    echo '<td>Acima de R$ ' . number_format($tier['gmv_min'], 0, ',', '.') . '</td>';

                } else {

                    echo '<td>R$ ' . number_format($tier['gmv_min'], 0, ',', '.') . ' - ' . number_format($tier['gmv_max'], 0, ',', '.') . '</td>';

                }

                echo '<td>' . esc_html($tier['fees']['physical']) . '%</td>';

                echo '<td>' . esc_html($tier['fees']['digital']) . '%</td>';

                echo '<td>' . esc_html($tier['fees']['subscription']) . '%</td>';

                echo '<td>' . esc_html($tier['fees']['quote'] ?? '-') . '%</td>';

                echo '<td>' . esc_html($tier['fees']['appointment'] ?? '-') . '%</td>';

                echo '</tr>';

            }

        }

        echo '</tbody>';

        echo '</table>';

        echo '</div>';

        

        echo '</div>'; // fees-card-content

        echo '</div>'; // hng-fees-info-card



        // Filter/Category buttons

        echo '<div class="hng-gateway-filters">';

        echo '<button class="filter-btn active" data-filter="all">Todos (13)</button>';

        echo '<button class="filter-btn" data-filter="fintech">Fintech (6)</button>';

        echo '<button class="filter-btn" data-filter="banks">Bancos (5)</button>';

        echo '<button class="filter-btn" data-filter="marketplace">Marketplace (2)</button>';

        echo '</div>';



        echo '<div class="hng-gateways-grid">';

        $gateways = self::get_gateways();

        foreach ($gateways as $id => $gw) {

            $enabled = get_option('hng_gateway_' . $id . '_enabled', 'no') === 'yes';

            $category = isset($gw['category']) ? $gw['category'] : 'other';

            

            echo '<div class="hng-gateway-item" data-gateway="' . esc_attr($id) . '" data-category="' . esc_attr($category) . '">';

            

            // Card header with logo/icon

            echo '<div class="gateway-header">';

            echo '<div class="gateway-icon ' . esc_attr($id) . '">';

            echo '<span class="dashicons dashicons-' . esc_attr($gw['icon']) . '"></span>';

            echo '</div>';

            echo '<div class="gateway-name-section">';

            echo '<div class="gateway-name-with-status">';

            echo '<h3>' . esc_html($gw['name']) . '</h3>';

            echo '<span class="gateway-api-status" data-gateway="' . esc_attr($id) . '" title="Clique em testar para verificar o status">';

            echo '<span class="status-dot status-red"></span>';

            echo '<span class="status-text">Não testado</span>';

            echo '</span>';

            echo '</div>';

            echo '<p class="gateway-category">' . esc_html(self::get_category_label($category)) . '</p>';

            echo '</div>';



            echo '</div>';



            // Description

            echo '<div class="gateway-description">';

            echo '<p>' . esc_html($gw['description']) . '</p>';

            echo '</div>';



            // Methods & Fees

            echo '<div class="gateway-details">';

            

            if (!empty($gw['methods'])) {

                echo '<div class="detail-group">';

                echo '<label class="detail-label">Métodos</label>';

                echo '<div class="detail-badges">';

                foreach ($gw['methods'] as $m) {

                    echo '<span class="badge badge-method">' . esc_html($m) . '</span>';

                }

                echo '</div>';

                echo '</div>';

            }

            

            if (!empty($gw['fees'])) {

                echo '<div class="detail-group">';

                echo '<label class="detail-label">Taxas</label>';

                echo '<div class="detail-badges">';

                foreach ($gw['fees'] as $f) {

                    echo '<span class="badge badge-fee">' . esc_html($f) . '</span>';

                }

                echo '</div>';

                echo '</div>';

            }

            

            echo '</div>';



            // Actions

            echo '<div class="gateway-actions">';

            

            // Toggle switch

            echo '<div class="action-toggle">';

            echo '<label class="hng-switch">';

            echo '<input type="checkbox" class="gateway-toggle" data-gateway="' . esc_attr($id) . '" ' . checked($enabled, true, false) . ' />';

            echo '<span class="hng-switch-slider"></span>';

            echo '</label>';

            echo '<span class="toggle-label">' . ($enabled ? 'Ativo' : 'Inativo') . '</span>';

            echo '</div>';

            

            // Buttons

            echo '<div class="action-buttons">';

            echo '<button type="button" class="button button-secondary hng-toggle-config" data-gateway="' . esc_attr($id) . '" title="Configurar credenciais">';

            echo '<span class="dashicons dashicons-admin-tools"></span>';

            echo '</button>';

            echo '<button type="button" class="button button-secondary gateway-test-btn" data-gateway="' . esc_attr($id) . '" title="Testar conexao">';

            echo '<span class="dashicons dashicons-update"></span>';

            echo '</button>';

            echo '<a href="' . esc_url('#') . '" target="_blank" class="button button-link gateway-docs-btn" title="Ver documentacao">';

            echo '<span class="dashicons dashicons-external"></span>';

            echo '</a>';

            echo '</div>';

            

            echo '</div>';



            // Config form (initially hidden)

            echo '<div class="hng-gateway-config-wrapper" data-gateway="' . esc_attr($id) . '" style="display:none;">';

            self::render_gateway_form($id);

            echo '</div>';

            

            echo '</div>';

        }

        echo '</div>';



        // Enqueue gateway management scripts

        wp_enqueue_script(

            'hng-admin-gateways',

            HNG_COMMERCE_URL . 'assets/js/admin-gateways.js',

            array('jquery'),

            HNG_COMMERCE_VERSION,

            true

        );

        

        wp_localize_script('hng-admin-gateways', 'hngGatewaysPage', array(

            'ajaxUrl' => admin_url('admin-ajax.php'),

            'nonce' => $nonce,

        ));



        // Enhanced inline CSS for the gateways page

        echo '<style>

        .hng-gateways-page {

            background: #f5f5f5;

            padding: 20px 0;

        }

        

        /* Card de Taxas HNG Commerce */

        .hng-fees-info-card {

            background: linear-gradient(135deg, #f8f9ff 0%, #fff 100%);

            border: 1px solid #e0e6ff;

            border-radius: 12px;

            margin: 0 20px 25px 20px;

            overflow: hidden;

            box-shadow: 0 2px 8px rgba(0,0,0,0.05);

        }

        

        .fees-card-header {

            display: flex;

            justify-content: space-between;

            align-items: center;

            padding: 16px 20px;

            background: linear-gradient(135deg, #0073aa 0%, #005a87 100%);

            color: white;

            cursor: pointer;

        }

        

        .fees-card-title {

            display: flex;

            align-items: center;

            gap: 10px;

            font-size: 15px;

        }

        

        .fees-card-title .dashicons {

            font-size: 20px;

            width: 20px;

            height: 20px;

        }

        

        .fees-card-toggle .dashicons {

            transition: transform 0.3s ease;

        }

        

        .fees-card-content {

            padding: 20px;

        }

        

        .fees-card-content.collapsed {

            display: none;

        }

        

        .fees-description {

            background: #fff;

            padding: 15px;

            border-radius: 8px;

            border-left: 4px solid #0073aa;

            margin-bottom: 20px;

        }

        

        .fees-description p {

            margin: 0 0 10px 0;

            color: #333;

        }

        

        .fees-description ul {

            margin: 0;

            padding-left: 20px;

        }

        

        .fees-description li {

            margin: 5px 0;

            color: #555;

        }

        

        .fees-tier-info {

            display: flex;

            gap: 20px;

            align-items: flex-start;

            background: #fff;

            padding: 20px;

            border-radius: 8px;

            margin-bottom: 20px;

            border: 1px solid #eee;

        }

        

        .tier-badge {

            display: flex;

            flex-direction: column;

            align-items: center;

            padding: 15px 20px;

            border-radius: 10px;

            color: white;

            min-width: 100px;

            text-align: center;

        }

        

        .tier-icon {

            font-size: 24px;

            margin-bottom: 5px;

        }

        

        .tier-name {

            font-weight: bold;

            font-size: 14px;

        }

        

        .tier-details {

            flex: 1;

        }

        

        .tier-gmv {

            margin: 0 0 10px 0;

            color: #333;

        }

        

        .tier-fees-list {

            display: flex;

            gap: 15px;

            flex-wrap: wrap;

        }

        

        .fee-item {

            background: #f5f5f5;

            padding: 8px 14px;

            border-radius: 6px;

            font-size: 13px;

            color: #444;

        }

        

        .tier-minimum {

            margin: 10px 0 0 0;

            color: #888;

        }

        

        .fees-next-tier {

            background: linear-gradient(90deg, #e8f5e9 0%, #f1f8e9 100%);

            padding: 15px 20px;

            border-radius: 8px;

            margin-bottom: 20px;

            border: 1px solid #c8e6c9;

        }

        

        .next-tier-msg {

            margin: 0 0 12px 0;

            color: #2e7d32;

        }

        

        .next-tier-progress {

            background: #ddd;

            height: 10px;

            border-radius: 5px;

            overflow: hidden;

        }

        

        .next-tier-progress .progress-bar {

            height: 100%;

            background: linear-gradient(90deg, #4caf50 0%, #81c784 100%);

            border-radius: 5px;

            transition: width 0.5s ease;

        }

        

        .fees-tiers-table {

            background: #fff;

            padding: 20px;

            border-radius: 8px;

            border: 1px solid #eee;

        }

        

        .fees-tiers-table h4 {

            margin: 0 0 15px 0;

            color: #333;

            font-size: 14px;

        }

        

        .fees-tiers-table table {

            width: 100%;

            border-collapse: collapse;

            font-size: 13px;

        }

        

        .fees-tiers-table th,

        .fees-tiers-table td {

            padding: 10px 12px;

            text-align: left;

            border-bottom: 1px solid #eee;

        }

        

        .fees-tiers-table th {

            background: #f8f9fa;

            font-weight: 600;

            color: #333;

        }

        

        .fees-tiers-table tr.current-tier {

            background: #e3f2fd;

        }

        

        .fees-tiers-table tr.current-tier td {

            font-weight: 500;

        }

        

        .tier-name-badge {

            display: inline-block;

            padding: 4px 10px;

            border-radius: 4px;

            color: white;

            font-size: 12px;

            font-weight: 500;

        }

        

        @media (max-width: 768px) {

            .fees-tier-info {

                flex-direction: column;

            }

            .tier-fees-list {

                flex-direction: column;

                gap: 8px;

            }

        }

        

        .hng-page-header {

            display: flex;

            justify-content: space-between;

            align-items: flex-start;

            margin-bottom: 30px;

            padding: 0 20px;

        }

        

        .hng-page-header h1 {

            display: flex;

            align-items: center;

            gap: 12px;

            margin: 0;

            color: #222;

            font-size: 28px;

        }

        

        .hng-page-header .dashicons {

            color: #0073aa;

        }

        

        .hng-header-title {

            flex: 1;

        }

        

        .hng-header-title p {

            color: #666;

            margin: 8px 0 0 0;

        }

        

        .hng-header-actions {

            display: flex;

            gap: 10px;

        }

        

        .hng-gateway-filters {

            display: flex;

            gap: 10px;

            margin-bottom: 25px;

            padding: 0 20px;

            flex-wrap: wrap;

        }

        

        .filter-btn {

            padding: 8px 16px;

            border: 2px solid #ddd;

            background: white;

            border-radius: 20px;

            cursor: pointer;

            font-size: 13px;

            font-weight: 500;

            color: #666;

            transition: all 0.3s ease;

        }

        

        .filter-btn:hover {

            border-color: #0073aa;

            color: #0073aa;

        }

        

        .filter-btn.active {

            background: #0073aa;

            border-color: #0073aa;

            color: white;

        }

        

        .hng-gateways-grid {

            display: grid;

            grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));

            gap: 20px;

            padding: 0 20px;

        }

        

        .hng-gateway-item {

            background: white;

            border-radius: 12px;

            overflow: hidden;

            box-shadow: 0 2px 8px rgba(0,0,0,0.08);

            transition: all 0.3s ease;

            position: relative;

            display: flex;

            flex-direction: column;

            border: 2px solid transparent;

        }

        

        .hng-gateway-item:hover {

            box-shadow: 0 4px 16px rgba(0,0,0,0.12);

            border-color: #0073aa;

            transform: translateY(-2px);

        }

        

        .gateway-name-with-status {

            display: flex;

            align-items: center;

            justify-content: space-between;

            gap: 10px;

        }

        

        .gateway-api-status {

            display: flex;

            align-items: center;

            gap: 6px;

            font-size: 12px;

            white-space: nowrap;

        }

        

        .status-dot {

            width: 10px;

            height: 10px;

            border-radius: 50%;

            display: inline-block;

            box-shadow: 0 0 0 1px rgba(0,0,0,0.1);

        }

        

        .status-dot.status-unknown {

            background: #6c757d;

            animation: pulse 1s infinite;

        }

        

        .status-dot.status-green {

            background: #28a745;

        }

        

        .status-dot.status-yellow {

            background: #ffc107;

        }

        

        .status-dot.status-red {

            background: #dc3545;

        }

        

        @keyframes pulse {

            0% { opacity: 1; }

            50% { opacity: 0.5; }

            100% { opacity: 1; }

        }

        

        .status-text {

            color: #666;

        }

        

        .gateway-header {

            display: flex;

            align-items: center;

            gap: 12px;

            padding: 16px;

            border-bottom: 1px solid #f0f0f0;

        }

        

        .gateway-icon {

            width: 48px;

            height: 48px;

            display: flex;

            align-items: center;

            justify-content: center;

            background: #f5f5f5;

            border-radius: 8px;

            font-size: 24px;

            color: #0073aa;

            flex-shrink: 0;

        }

        

        .gateway-name-section h3 {

            margin: 0;

            color: #222;

            font-size: 16px;

            font-weight: 600;

        }

        

        .gateway-category {

            margin: 4px 0 0 0;

            color: #999;

            font-size: 12px;

            text-transform: uppercase;

            letter-spacing: 0.5px;

        }

        

        .gateway-description {

            padding: 12px 16px;

            border-bottom: 1px solid #f0f0f0;

        }

        

        .gateway-description p {

            margin: 0;

            color: #666;

            font-size: 13px;

        }

        

        .gateway-details {

            flex: 1;

            padding: 16px;

        }

        

        .detail-group {

            margin-bottom: 12px;

        }

        

        .detail-group:last-child {

            margin-bottom: 0;

        }

        

        .detail-label {

            display: block;

            font-size: 11px;

            font-weight: 600;

            color: #999;

            text-transform: uppercase;

            margin-bottom: 6px;

            letter-spacing: 0.5px;

        }

        

        .detail-badges {

            display: flex;

            flex-wrap: wrap;

            gap: 6px;

        }

        

        .badge {

            display: inline-block;

            padding: 4px 10px;

            border-radius: 4px;

            font-size: 12px;

            font-weight: 500;

        }

        

        .badge-method {

            background: #e7f3ff;

            color: #0073aa;

        }

        

        .badge-fee {

            background: #f0f9ff;

            color: #0066cc;

        }

        

        .gateway-actions {

            padding: 16px;

            border-top: 1px solid #f0f0f0;

            display: flex;

            align-items: center;

            justify-content: space-between;

            gap: 8px;

        }

        

        .action-toggle {

            display: flex;

            align-items: center;

            gap: 8px;

            flex-shrink: 0;

        }

        

        .toggle-label {

            font-size: 12px;

            font-weight: 500;

            color: #666;

            min-width: 50px;

        }

        

        .action-buttons {

            display: flex;

            gap: 6px;

        }

        

        .action-buttons .button {

            padding: 6px 8px;

            height: auto;

            min-width: 36px;

            display: flex;

            align-items: center;

            justify-content: center;

        }

        

        .action-buttons .dashicons {

            font-size: 16px;

            width: 16px;

            height: 16px;

            margin: 0;

        }

        

        .gateway-docs-btn {

            text-decoration: none;

            color: #0073aa;

            padding: 6px 8px;

        }

        

        .gateway-docs-btn:hover {

            color: #005a87;

        }

        

        .hng-switch {

            position: relative;

            display: inline-block;

            width: 42px;

            height: 24px;

        }

        

        .hng-switch input {

            opacity: 0;

            width: 0;

            height: 0;

        }

        

        .hng-switch-slider {

            position: absolute;

            cursor: pointer;

            top: 0;

            left: 0;

            right: 0;

            bottom: 0;

            background: #ccc;

            transition: 0.3s;

            border-radius: 24px;

        }

        

        .hng-switch-slider:before {

            position: absolute;

            content: "";

            height: 18px;

            width: 18px;

            left: 3px;

            bottom: 3px;

            background: white;

            transition: 0.3s;

            border-radius: 50%;

            box-shadow: 0 2px 4px rgba(0,0,0,0.1);

        }

        

        .hng-switch input:checked + .hng-switch-slider {

            background: #28a745;

        }

        

        .hng-switch input:checked + .hng-switch-slider:before {

            transform: translateX(18px);

        }

        

        .hng-gateway-config-wrapper {

            margin-top: 12px;

            padding: 16px;

            border-top: 1px solid #ddd;

            background: #fafafa;

            display: none;

        }

        

        .hng-gateway-config-wrapper.show {

            display: block;

        }

        

        @media (max-width: 768px) {

            .hng-page-header {

                flex-direction: column;

                gap: 15px;

            }

            

            .hng-gateways-grid {

                grid-template-columns: 1fr;

            }

        }



        /* Notification System */

        .hng-notification {

            position: fixed;

            top: 20px;

            right: 20px;

            padding: 16px 20px;

            background: white;

            border-radius: 8px;

            box-shadow: 0 4px 12px rgba(0,0,0,0.15);

            z-index: 9999;

            min-width: 300px;

            max-width: 450px;

            animation: slideIn 0.3s ease;

            display: flex;

            align-items: center;

            gap: 12px;

            border-left: 4px solid #0073aa;

        }

        

        .hng-notification.success {

            border-left-color: #28a745;

        }

        

        .hng-notification.error {

            border-left-color: #dc3545;

        }

        

        .hng-notification.warning {

            border-left-color: #ffc107;

        }

        

        .hng-notification.info {

            border-left-color: #0073aa;

        }

        

        .hng-notification-icon {

            font-size: 20px;

            flex-shrink: 0;

            display: flex;

            align-items: center;

            justify-content: center;

            width: 24px;

        }

        

        .hng-notification.success .hng-notification-icon {

            color: #28a745;

        }

        

        .hng-notification.error .hng-notification-icon {

            color: #dc3545;

        }

        

        .hng-notification.warning .hng-notification-icon {

            color: #ffc107;

        }

        

        .hng-notification.info .hng-notification-icon {

            color: #0073aa;

        }

        

        .hng-notification-content {

            flex: 1;

            color: #333;

            font-size: 14px;

            font-weight: 500;

        }

        

        .hng-notification-close {

            flex-shrink: 0;

            cursor: pointer;

            color: #999;

            font-size: 18px;

            line-height: 1;

            padding: 0 0 0 10px;

            transition: color 0.2s ease;

        }

        

        .hng-notification-close:hover {

            color: #333;

        }

        

        @keyframes slideIn {

            from {

                transform: translateX(400px);

                opacity: 0;

            }

            to {

                transform: translateX(0);

                opacity: 1;

            }

        }

        

        @keyframes slideOut {

            from {

                transform: translateX(0);

                opacity: 1;

            }

            to {

                transform: translateX(400px);

                opacity: 0;

            }

        }

        

        .hng-notification.removing {

            animation: slideOut 0.3s ease;

        }

        </style>';



        echo '</div>';

    }

    

    /**

     * List of available gateways

     */

    private static function get_gateways() {

        return [

            'asaas' => [

                'name' => 'Asaas',

                'description' => 'PIX, Boleto e Cartao de Credito',

                'icon' => 'money-alt',

                'category' => 'fintech',

                'methods' => ['PIX', 'Boleto', 'Cartao'],

                'fees' => ['PIX: 0,99% + R$0,49', 'Boleto: R$3,49', 'Cartao: 2,99%']

            ],

            'mercadopago' => [

                'name' => 'Mercado Pago',

                'description' => 'Solucao completa de pagamentos',

                'icon' => 'cart',

                'category' => 'marketplace',

                'methods' => ['PIX', 'Boleto', 'Cartao', 'Checkout Pro'],

                'fees' => ['PIX: 0,99%', 'Boleto: R$3,79', 'Cartao: 4,99%']

            ],

            'pagseguro' => [

                'name' => 'PagSeguro',

                'description' => 'Gateway UOL PagSeguro',

                'icon' => 'shield',

                'category' => 'marketplace',

                'methods' => ['PIX', 'Boleto', 'Cartao'],

                'fees' => ['PIX: 0,99%', 'Boleto: R$3,00', 'Cartao: 3,99%']

            ],

            'pagarme' => [

                'name' => 'Pagar.me',

                'description' => 'PIX, Boleto, Cartao e Split Payment',

                'icon' => 'admin-generic',

                'category' => 'fintech',

                'methods' => ['PIX', 'Boleto', 'Cartao', 'Split'],

                'fees' => ['PIX: 0,99%', 'Boleto: R$3,49', 'Cartao: 3,79%']

            ],

            'nubank' => [

                'name' => 'Nubank / NuPay',

                'description' => 'PIX e Cartao de Credito',

                'icon' => 'star-filled',

                'category' => 'fintech',

                'methods' => ['PIX', 'Cartao'],

                'fees' => ['PIX: 0,50%', 'Cartao: 2,99%']

            ],

            'inter' => [

                'name' => 'Banco Inter',

                'description' => 'PIX com OAuth2',

                'icon' => 'building',

                'category' => 'banks',

                'methods' => ['PIX'],

                'fees' => ['PIX: 0,49%']

            ],

            'c6bank' => [

                'name' => 'C6 Bank',

                'description' => 'Pagamentos via C6 Bank',

                'icon' => 'building',

                'category' => 'banks',

                'methods' => ['PIX'],

                'fees' => ['PIX: 0,50%']

            ],

            'bb' => [

                'name' => 'Banco do Brasil',

                'description' => 'PIX institucional',

                'icon' => 'building',

                'category' => 'banks',

                'methods' => ['PIX'],

                'fees' => ['PIX: 0,99%']

            ],

            'itau' => [

                'name' => 'Itau',

                'description' => 'Gateway Itau',

                'icon' => 'building',

                'category' => 'banks',

                'methods' => ['PIX'],

                'fees' => ['PIX: 0,99%']

            ],

            'bradesco' => [

                'name' => 'Bradesco',

                'description' => 'Gateway Bradesco',

                'icon' => 'building',

                'category' => 'banks',

                'methods' => ['PIX'],

                'fees' => ['PIX: 0,99%']

            ],

            'santander' => [

                'name' => 'Santander',

                'description' => 'Gateway Santander',

                'icon' => 'building',

                'category' => 'banks',

                'methods' => ['PIX'],

                'fees' => ['PIX: 0,99%']

            ],

            'stripe' => [

                'name' => 'Stripe',

                'description' => 'Cartao de Credito Internacional',

                'icon' => 'credit',

                'category' => 'fintech',

                'methods' => ['Cartao Credito', 'Cartao Debito', 'Apple Pay', 'Google Pay'],

                'fees' => ['Cartao: 2,90% + $0,30', 'Internacional: 3,90% + $0,30']

            ],

            'paypal' => [

                'name' => 'PayPal',

                'description' => 'Pagamentos internacionais',

                'icon' => 'money-alt',

                'category' => 'fintech',

                'methods' => ['PayPal Wallet', 'Cartao'],

                'fees' => ['PayPal: 4,99% + $0,49', 'Cartao: 3,99% + $0,49']

            ],

            'picpay' => [

                'name' => 'PicPay',

                'description' => 'Carteira digital brasileira',

                'icon' => 'admin-generic',

                'category' => 'fintech',

                'methods' => ['PicPay Wallet', 'QR Code'],

                'fees' => ['PicPay: 1,99%']

            ],

            'cielo' => [

                'name' => 'Cielo',

                'description' => 'Gateway Cielo de pagamentos',

                'icon' => 'building',

                'category' => 'marketplace',

                'methods' => ['PIX', 'Cartao', 'Boleto'],

                'fees' => ['PIX: 0,99%', 'Cartao: 2,75%', 'Boleto: R$2,80']

            ],

            'rede' => [

                'name' => 'Rede',

                'description' => 'Processadora Rede',

                'icon' => 'building',

                'category' => 'marketplace',

                'methods' => ['PIX', 'Cartao', 'Boleto'],

                'fees' => ['PIX: 0,99%', 'Cartao: 2,90%', 'Boleto: R$3,00']

            ],

            'getnet' => [

                'name' => 'GetNet',

                'description' => 'Gateway GetNet Santander',

                'icon' => 'building',

                'category' => 'marketplace',

                'methods' => ['PIX', 'Cartao', 'Boleto'],

                'fees' => ['PIX: 0,99%', 'Cartao: 2,95%', 'Boleto: R$2,90']

            ],

            'stone' => [

                'name' => 'Stone',

                'description' => 'Adquirente Stone',

                'icon' => 'building',

                'category' => 'marketplace',

                'methods' => ['PIX', 'Cartao'],

                'fees' => ['PIX: 0,99%', 'Cartao: 2,85%']

            ]

        ];

    }

    

    /**

     * Get category label

     */

    private static function get_category_label($category) {

        $labels = [

            'fintech' => 'Fintech',

            'banks' => 'Banco',

            'marketplace' => 'Marketplace',

            'other' => 'Outro'

        ];

        return isset($labels[$category]) ? $labels[$category] : $labels['other'];

    }

    

    /**

     * Renderiza formulario de configuracao para cada gateway

     */

    private static function render_gateway_form($gatewayId) {

        $configs = [];

        $all_gateways = self::get_gateways();

        $gateway = isset($all_gateways[$gatewayId]) ? $all_gateways[$gatewayId] : ['name' => $gatewayId, 'description' => ''];

        

        if ($gatewayId === 'asaas') {

            $configs = [

                'api_key' => get_option('hng_asaas_api_key', ''),

                'sandbox' => get_option('hng_asaas_sandbox', 1),

                'wallet_id' => get_option('hng_asaas_wallet_id', '')

            ];

            echo '<h3 style="margin-top: 0;">Config Asaas</h3>';

            echo '<form class="gateway-config-form-inner" data-gateway="asaas">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_asaas', false);

            echo '<p><label><strong>API Key:</strong></label><br>';

            echo '<input type="password" name="api_key" class="regular-text" value="' . esc_attr($configs['api_key']) . '" placeholder="$aact_...">';

            echo '<br><small>Obtenha em painel Asaas</small></p>';

            echo '<p><label><strong>Wallet ID (Split):</strong></label><br>';

            echo '<input type="text" name="wallet_id" class="regular-text" value="' . esc_attr($configs['wallet_id']) . '" placeholder="ID da carteira (opcional)"></p>';

            echo '<p><label><input type="checkbox" name="sandbox" value="1" ' . checked($configs['sandbox'], 1, false) . '> Modo Sandbox (Teste)</label>';

            echo '<br><small>Desative em producao!</small></p>';

            $adv_enabled = get_option('hng_asaas_advanced_integration', 'no') === 'yes';

            echo '<p style="display:flex;align-items:center;gap:8px;"><strong>Integração avançada:</strong> ';

            echo '<label class="hng-switch" style="margin-left:8px;">';

            echo '<input type="checkbox" class="advanced-toggle" data-gateway="asaas" ' . checked($adv_enabled, true, false) . ' />';

            echo '<span class="hng-switch-slider"></span>';

            echo '</label>';

            echo '<span class="description">Ativa recursos de sincronização e webhooks.</span>';

            echo '</p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        } elseif ($gatewayId === 'mercadopago') {

            $configs = [

                'access_token' => get_option('hng_mercadopago_access_token', ''),

                'public_key' => get_option('hng_mercadopago_public_key', ''),

                'sandbox' => get_option('hng_mercadopago_sandbox', 1)

            ];

            echo '<h3 style="margin-top: 0;">Config Mercado Pago</h3>';

            echo '<form class="gateway-config-form-inner" data-gateway="mercadopago">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_mercadopago', false);

            echo '<p><label><strong>Access Token:</strong></label><br>';

            echo '<input type="password" name="access_token" class="regular-text" value="' . esc_attr($configs['access_token']) . '" placeholder="APP_USR-...">';

            echo '<br><small>Obtenha em Mercado Pago Developers</small></p>';

            echo '<p><label><strong>Public Key:</strong></label><br>';

            echo '<input type="text" name="public_key" class="regular-text" value="' . esc_attr($configs['public_key']) . '" placeholder="APP_USR-... (publico)"></p>';

            echo '<p><label><input type="checkbox" name="sandbox" value="1" ' . checked($configs['sandbox'], 1, false) . '> Modo Sandbox (Teste)</label>';

            echo '<br><small>Desative em producao!</small></p>';

            $adv_enabled = get_option('hng_mercadopago_advanced_integration', 'no') === 'yes';

            echo '<p style="display:flex;align-items:center;gap:8px;"><strong>Integração avançada:</strong> ';

            echo '<label class="hng-switch" style="margin-left:8px;">';

            echo '<input type="checkbox" class="advanced-toggle" data-gateway="mercadopago" ' . checked($adv_enabled, true, false) . ' />';

            echo '<span class="hng-switch-slider"></span>';

            echo '</label>';

            echo '<span class="description">Ativa clientes, assinaturas e webhooks.</span>';

            echo '</p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        } elseif ($gatewayId === 'pagseguro') {

            $configs = [

                'token' => get_option('hng_ps_token', ''),

                'email' => get_option('hng_ps_email', ''),

                'sandbox' => get_option('hng_ps_sandbox', 'yes') === 'yes'

            ];

            echo '<h3 style="margin-top: 0;">Config PagSeguro</h3>';

            echo '<form class="gateway-config-form-inner" data-gateway="pagseguro">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_pagseguro', false);

            echo '<p><label><strong>Token / API Key:</strong></label><br>';

            echo '<input type="password" name="token" class="regular-text" value="' . esc_attr($configs['token']) . '" placeholder="token_pagseguro..."></p>';

            echo '<p><label><strong>Email de conta:</strong></label><br>';

            echo '<input type="email" name="email" class="regular-text" value="' . esc_attr($configs['email']) . '" placeholder="seu@email.com"></p>';

            echo '<p><label><input type="checkbox" name="sandbox" value="1" ' . checked($configs['sandbox'], true, false) . '> Modo Sandbox (Teste)</label>';

            echo '<br><small>Obtenha seu token no Portal PagSeguro</small></p>';

            $adv_enabled = get_option('hng_pagseguro_advanced_integration', 'no') === 'yes';

            echo '<p style="display:flex;align-items:center;gap:8px;"><strong>Integração avançada:</strong> ';

            echo '<label class="hng-switch" style="margin-left:8px;">';

            echo '<input type="checkbox" class="advanced-toggle" data-gateway="pagseguro" ' . checked($adv_enabled, true, false) . ' />';

            echo '<span class="hng-switch-slider"></span>';

            echo '</label>';

            echo '<span class="description">Ativa assinaturas, clientes e faturamento.</span>';

            echo '</p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        } elseif ($gatewayId === 'pagarme') {

            $configs = [

                'secret_key' => get_option('hng_pagarme_secret_key', ''),

                'public_key' => get_option('hng_pagarme_public_key', ''),

                'enabled' => get_option('hng_pagarme_enabled', 'no') === 'yes'

            ];

            echo '<h3 style="margin-top: 0;">Config Pagar.me</h3>';

            echo '<form class="gateway-config-form-inner" data-gateway="pagarme">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_pagarme', false);

            echo '<p><label><strong>Secret Key:</strong></label><br>';

            echo '<input type="password" name="secret_key" class="regular-text" value="' . esc_attr($configs['secret_key']) . '" placeholder="sk_live_..."></p>';

            echo '<p><label><strong>Public Key:</strong></label><br>';

            echo '<input type="text" name="public_key" class="regular-text" value="' . esc_attr($configs['public_key']) . '" placeholder="pk_live_..."></p>';

            echo '<p><label><input type="checkbox" name="enabled" value="1" ' . checked($configs['enabled'], true, false) . '> Habilitar gateway</label></p>';

            $adv_enabled = get_option('hng_pagarme_advanced_integration', 'no') === 'yes';

            echo '<p style="display:flex;align-items:center;gap:8px;"><strong>Integração avançada:</strong> ';

            echo '<label class="hng-switch" style="margin-left:8px;">';

            echo '<input type="checkbox" class="advanced-toggle" data-gateway="pagarme" ' . checked($adv_enabled, true, false) . ' />';

            echo '<span class="hng-switch-slider"></span>';

            echo '</label>';

            echo '<span class="description">Ativa Split Payment e assinaturas recorrentes.</span>';

            echo '</p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        } elseif ($gatewayId === 'nubank') {

            $configs = [

                'client_id' => get_option('hng_nubank_client_id', ''),

                'client_secret' => get_option('hng_nubank_client_secret', ''),

                'cert_path' => get_option('hng_nubank_cert_path', ''),

                'environment' => get_option('hng_nubank_environment', 'sandbox')

            ];

            echo '<h3 style="margin-top: 0;">Config Nubank / NuPay</h3>';

            echo '<form class="gateway-config-form-inner" data-gateway="nubank">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_nubank', false);

            echo '<p><label><strong>Client ID:</strong></label><br>';

            echo '<input type="text" name="client_id" class="regular-text" value="' . esc_attr($configs['client_id']) . '" placeholder="Client ID da aplicação"></p>';

            echo '<p><label><strong>Client Secret:</strong></label><br>';

            echo '<input type="password" name="client_secret" class="regular-text" value="' . esc_attr($configs['client_secret']) . '" placeholder="Client Secret"></p>';

            echo '<p><label><strong>Caminho Certificado (.pem):</strong></label><br>';

            echo '<input type="text" name="cert_path" class="regular-text" value="' . esc_attr($configs['cert_path']) . '" placeholder="/path/to/cert.pem"><br><small>Certificado digital do Portal do Desenvolvedor</small></p>';

            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';

            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';

            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';

            echo '</select></p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        } elseif ($gatewayId === 'inter') {

            $configs = [

                'client_id' => get_option('hng_inter_client_id', ''),

                'client_secret' => get_option('hng_inter_client_secret', ''),

                'conta_corrente' => get_option('hng_inter_conta_corrente', ''),

                'cert_path' => get_option('hng_inter_cert_path', ''),

                'key_path' => get_option('hng_inter_key_path', ''),

                'environment' => get_option('hng_inter_environment', 'sandbox')

            ];

            echo '<h3 style="margin-top: 0;">Config Banco Inter</h3>';

            echo '<form class="gateway-config-form-inner" data-gateway="inter">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_inter', false);

            echo '<p><label><strong>Client ID:</strong></label><br>';

            echo '<input type="text" name="client_id" class="regular-text" value="' . esc_attr($configs['client_id']) . '" placeholder="Client ID OAuth"></p>';

            echo '<p><label><strong>Client Secret:</strong></label><br>';

            echo '<input type="password" name="client_secret" class="regular-text" value="' . esc_attr($configs['client_secret']) . '" placeholder="Client Secret"></p>';

            echo '<p><label><strong>Conta Corrente:</strong></label><br>';

            echo '<input type="text" name="conta_corrente" class="regular-text" value="' . esc_attr($configs['conta_corrente']) . '" placeholder="Número da conta (sem dígito)"></p>';

            echo '<p><label><strong>Caminho Certificado (.crt):</strong></label><br>';

            echo '<input type="text" name="cert_path" class="regular-text" value="' . esc_attr($configs['cert_path']) . '" placeholder="/path/to/cert.crt"></p>';

            echo '<p><label><strong>Caminho Chave (.key):</strong></label><br>';

            echo '<input type="text" name="key_path" class="regular-text" value="' . esc_attr($configs['key_path']) . '" placeholder="/path/to/key.key"></p>';

            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';

            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';

            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';

            echo '</select></p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        } elseif ($gatewayId === 'c6bank') {

            $configs = [

                'client_id' => get_option('hng_c6bank_client_id', ''),

                'client_secret' => get_option('hng_c6bank_client_secret', ''),

                'pix_key' => get_option('hng_c6bank_pix_key', ''),

                'environment' => get_option('hng_c6bank_environment', 'sandbox')

            ];

            echo '<h3 style="margin-top: 0;">Config C6 Bank</h3>';

            echo '<form class="gateway-config-form-inner" data-gateway="c6bank">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_c6bank', false);

            echo '<p><label><strong>Client ID:</strong></label><br>';

            echo '<input type="text" name="client_id" class="regular-text" value="' . esc_attr($configs['client_id']) . '" placeholder="Client ID da aplicação"></p>';

            echo '<p><label><strong>Client Secret:</strong></label><br>';

            echo '<input type="password" name="client_secret" class="regular-text" value="' . esc_attr($configs['client_secret']) . '" placeholder="Client Secret"></p>';

            echo '<p><label><strong>Chave PIX:</strong></label><br>';

            echo '<input type="text" name="pix_key" class="regular-text" value="' . esc_attr($configs['pix_key']) . '" placeholder="Chave PIX da conta"></p>';

            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';

            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';

            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';

            echo '</select></p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        } elseif ($gatewayId === 'bb') {

            $configs = [

                'developer_key' => get_option('hng_bb_developer_key', ''),

                'client_id' => get_option('hng_bb_client_id', ''),

                'client_secret' => get_option('hng_bb_client_secret', ''),

                'convenio' => get_option('hng_bb_convenio', ''),

                'agencia' => get_option('hng_bb_agencia', ''),

                'conta' => get_option('hng_bb_conta', ''),

                'environment' => get_option('hng_bb_environment', 'sandbox')

            ];

            echo '<h3 style="margin-top: 0;">Config Banco do Brasil</h3>';

            echo '<form class="gateway-config-form-inner" data-gateway="bb">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_bb', false);

            echo '<p><label><strong>Developer Application Key:</strong></label><br>';

            echo '<input type="password" name="developer_key" class="regular-text" value="' . esc_attr($configs['developer_key']) . '" placeholder="Chave do Portal Developers BB"></p>';

            echo '<p><label><strong>Client ID:</strong></label><br>';

            echo '<input type="text" name="client_id" class="regular-text" value="' . esc_attr($configs['client_id']) . '" placeholder="Client ID OAuth"></p>';

            echo '<p><label><strong>Client Secret:</strong></label><br>';

            echo '<input type="password" name="client_secret" class="regular-text" value="' . esc_attr($configs['client_secret']) . '" placeholder="Client Secret"></p>';

            echo '<p><label><strong>Convênio:</strong></label><br>';

            echo '<input type="text" name="convenio" class="regular-text" value="' . esc_attr($configs['convenio']) . '" placeholder="7 dígitos"></p>';

            echo '<p><label><strong>Agência:</strong></label><br>';

            echo '<input type="text" name="agencia" class="regular-text" value="' . esc_attr($configs['agencia']) . '" placeholder="4 dígitos"></p>';

            echo '<p><label><strong>Conta Corrente:</strong></label><br>';

            echo '<input type="text" name="conta" class="regular-text" value="' . esc_attr($configs['conta']) . '" placeholder="Número da conta"></p>';

            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';

            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';

            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';

            echo '</select></p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        } elseif ($gatewayId === 'itau') {

            $configs = [

                'client_id' => get_option('hng_itau_client_id', ''),

                'client_secret' => get_option('hng_itau_client_secret', ''),

                'chave_pix' => get_option('hng_itau_chave_pix', ''),

                'beneficiario_id' => get_option('hng_itau_beneficiario_id', ''),

                'agencia' => get_option('hng_itau_agencia', ''),

                'conta' => get_option('hng_itau_conta', ''),

                'environment' => get_option('hng_itau_environment', 'sandbox')

            ];

            echo '<h3 style="margin-top: 0;">Config Itaú</h3>';

            echo '<form class="gateway-config-form-inner" data-gateway="itau">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_itau', false);

            echo '<p><label><strong>Client ID:</strong></label><br>';

            echo '<input type="text" name="client_id" class="regular-text" value="' . esc_attr($configs['client_id']) . '" placeholder="Client ID OAuth"></p>';

            echo '<p><label><strong>Client Secret:</strong></label><br>';

            echo '<input type="password" name="client_secret" class="regular-text" value="' . esc_attr($configs['client_secret']) . '" placeholder="Client Secret"></p>';

            echo '<p><label><strong>Chave PIX:</strong></label><br>';

            echo '<input type="text" name="chave_pix" class="regular-text" value="' . esc_attr($configs['chave_pix']) . '" placeholder="CPF/CNPJ/Email/Telefone"></p>';

            echo '<p><label><strong>ID Beneficiário:</strong></label><br>';

            echo '<input type="text" name="beneficiario_id" class="regular-text" value="' . esc_attr($configs['beneficiario_id']) . '" placeholder="ID no sistema Itaú"></p>';

            echo '<p><label><strong>Agência:</strong></label><br>';

            echo '<input type="text" name="agencia" class="regular-text" value="' . esc_attr($configs['agencia']) . '" placeholder="4 dígitos"></p>';

            echo '<p><label><strong>Conta Corrente:</strong></label><br>';

            echo '<input type="text" name="conta" class="regular-text" value="' . esc_attr($configs['conta']) . '" placeholder="Com dígito"></p>';

            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';

            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';

            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';

            echo '</select></p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        } elseif ($gatewayId === 'bradesco') {

            $configs = [

                'merchant_id' => get_option('hng_bradesco_merchant_id', ''),

                'merchant_key' => get_option('hng_bradesco_merchant_key', ''),

                'carteira' => get_option('hng_bradesco_carteira', ''),

                'agencia' => get_option('hng_bradesco_agencia', ''),

                'conta' => get_option('hng_bradesco_conta', ''),

                'environment' => get_option('hng_bradesco_environment', 'sandbox')

            ];

            echo '<h3 style="margin-top: 0;">Config Bradesco</h3>';

            echo '<form class="gateway-config-form-inner" data-gateway="bradesco">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_bradesco', false);

            echo '<p><label><strong>Merchant ID:</strong></label><br>';

            echo '<input type="text" name="merchant_id" class="regular-text" value="' . esc_attr($configs['merchant_id']) . '" placeholder="ID Shopfácil"></p>';

            echo '<p><label><strong>Merchant Key:</strong></label><br>';

            echo '<input type="password" name="merchant_key" class="regular-text" value="' . esc_attr($configs['merchant_key']) . '" placeholder="Chave de segurança"></p>';

            echo '<p><label><strong>Carteira:</strong></label><br>';

            echo '<input type="text" name="carteira" class="regular-text" value="' . esc_attr($configs['carteira']) . '" placeholder="Ex: 26"></p>';

            echo '<p><label><strong>Agência:</strong></label><br>';

            echo '<input type="text" name="agencia" class="regular-text" value="' . esc_attr($configs['agencia']) . '" placeholder="4 dígitos"></p>';

            echo '<p><label><strong>Conta Corrente:</strong></label><br>';

            echo '<input type="text" name="conta" class="regular-text" value="' . esc_attr($configs['conta']) . '" placeholder="Sem dígito"></p>';

            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';

            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';

            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';

            echo '</select></p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        } elseif ($gatewayId === 'santander') {

            $configs = [

                'seller_id' => get_option('hng_santander_seller_id', ''),

                'client_id' => get_option('hng_santander_client_id', ''),

                'client_secret' => get_option('hng_santander_client_secret', ''),

                'convenio' => get_option('hng_santander_convenio', ''),

                'carteira' => get_option('hng_santander_carteira', ''),

                'environment' => get_option('hng_santander_environment', 'sandbox')

            ];

            echo '<h3 style="margin-top: 0;">Config Santander</h3>';

            echo '<form class="gateway-config-form-inner" data-gateway="santander">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_santander', false);

            echo '<p><label><strong>Seller ID (Getnet):</strong></label><br>';

            echo '<input type="text" name="seller_id" class="regular-text" value="' . esc_attr($configs['seller_id']) . '" placeholder="ID do vendedor Getnet"></p>';

            echo '<p><label><strong>Client ID:</strong></label><br>';

            echo '<input type="text" name="client_id" class="regular-text" value="' . esc_attr($configs['client_id']) . '" placeholder="Client ID OAuth"></p>';

            echo '<p><label><strong>Client Secret:</strong></label><br>';

            echo '<input type="password" name="client_secret" class="regular-text" value="' . esc_attr($configs['client_secret']) . '" placeholder="Client Secret"></p>';

            echo '<p><label><strong>Convênio:</strong></label><br>';

            echo '<input type="text" name="convenio" class="regular-text" value="' . esc_attr($configs['convenio']) . '" placeholder="Código do convênio"></p>';

            echo '<p><label><strong>Carteira:</strong></label><br>';

            echo '<input type="text" name="carteira" class="regular-text" value="' . esc_attr($configs['carteira']) . '" placeholder="Ex: 102"></p>';

            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';

            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';

            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';

            echo '</select></p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        } elseif ($gatewayId === 'stripe') {
            $configs = [
                'secret_key' => get_option('hng_stripe_secret_key', ''),
                'publishable_key' => get_option('hng_stripe_publishable_key', ''),
                'webhook_secret' => get_option('hng_stripe_webhook_secret', '')
            ];
            echo '<h3 style="margin-top: 0;">Config Stripe</h3>';
            echo '<form class="gateway-config-form-inner" data-gateway="stripe">';
            wp_nonce_field('hng_save_gateway_config', '_wpnonce_stripe', false);
            echo '<p><label><strong>Secret Key:</strong></label><br>';
            echo '<input type="password" name="secret_key" class="regular-text" value="' . esc_attr($configs['secret_key']) . '" placeholder="sk_live_..."></p>';
            echo '<p><label><strong>Publishable Key:</strong></label><br>';
            echo '<input type="text" name="publishable_key" class="regular-text" value="' . esc_attr($configs['publishable_key']) . '" placeholder="pk_live_..."></p>';
            echo '<p><label><strong>Webhook Secret:</strong></label><br>';
            echo '<input type="password" name="webhook_secret" class="regular-text" value="' . esc_attr($configs['webhook_secret']) . '" placeholder="whsec_..."></p>';
            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';
            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';
            echo '<div class="test-result" style="margin-top: 10px;"></div>';
            echo '</form>';
        } elseif ($gatewayId === 'paypal') {
            $configs = [
                'client_id' => get_option('hng_paypal_client_id', ''),
                'client_secret' => get_option('hng_paypal_client_secret', ''),
                'webhook_token' => get_option('hng_paypal_webhook_token', ''),
                'environment' => get_option('hng_paypal_environment', 'sandbox')
            ];
            echo '<h3 style="margin-top: 0;">Config PayPal</h3>';
            echo '<form class="gateway-config-form-inner" data-gateway="paypal">';
            wp_nonce_field('hng_save_gateway_config', '_wpnonce_paypal', false);
            echo '<p><label><strong>Client ID:</strong></label><br>';
            echo '<input type="text" name="client_id" class="regular-text" value="' . esc_attr($configs['client_id']) . '" placeholder="Client ID"></p>';
            echo '<p><label><strong>Client Secret:</strong></label><br>';
            echo '<input type="password" name="client_secret" class="regular-text" value="' . esc_attr($configs['client_secret']) . '" placeholder="Client Secret"></p>';
            echo '<p><label><strong>Webhook Token:</strong></label><br>';
            echo '<input type="text" name="webhook_token" class="regular-text" value="' . esc_attr($configs['webhook_token']) . '" placeholder="Webhook ID"></p>';
            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';
            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';
            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';
            echo '</select></p>';
            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';
            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';
            echo '<div class="test-result" style="margin-top: 10px;"></div>';
            echo '</form>';
        } elseif ($gatewayId === 'picpay') {
            $configs = [
                'api_key' => get_option('hng_picpay_api_key', ''),
                'seller_token' => get_option('hng_picpay_seller_token', ''),
                'environment' => get_option('hng_picpay_environment', 'sandbox')
            ];
            echo '<h3 style="margin-top: 0;">Config PicPay</h3>';
            echo '<form class="gateway-config-form-inner" data-gateway="picpay">';
            wp_nonce_field('hng_save_gateway_config', '_wpnonce_picpay', false);
            echo '<p><label><strong>API Key:</strong></label><br>';
            echo '<input type="password" name="api_key" class="regular-text" value="' . esc_attr($configs['api_key']) . '" placeholder="API Key"></p>';
            echo '<p><label><strong>Seller Token:</strong></label><br>';
            echo '<input type="text" name="seller_token" class="regular-text" value="' . esc_attr($configs['seller_token']) . '" placeholder="Token do vendedor"></p>';
            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';
            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';
            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';
            echo '</select></p>';
            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';
            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';
            echo '<div class="test-result" style="margin-top: 10px;"></div>';
            echo '</form>';
        } elseif ($gatewayId === 'cielo') {
            $configs = [
                'merchant_id' => get_option('hng_cielo_merchant_id', ''),
                'api_key' => get_option('hng_cielo_api_key', ''),
                'webhook_secret' => get_option('hng_cielo_webhook_secret', ''),
                'environment' => get_option('hng_cielo_environment', 'sandbox')
            ];
            echo '<h3 style="margin-top: 0;">Config Cielo</h3>';
            echo '<form class="gateway-config-form-inner" data-gateway="cielo">';
            wp_nonce_field('hng_save_gateway_config', '_wpnonce_cielo', false);
            echo '<p><label><strong>Merchant ID:</strong></label><br>';
            echo '<input type="text" name="merchant_id" class="regular-text" value="' . esc_attr($configs['merchant_id']) . '" placeholder="Merchant ID Cielo"></p>';
            echo '<p><label><strong>API Key:</strong></label><br>';
            echo '<input type="password" name="api_key" class="regular-text" value="' . esc_attr($configs['api_key']) . '" placeholder="API Key"></p>';
            echo '<p><label><strong>Webhook Secret:</strong></label><br>';
            echo '<input type="password" name="webhook_secret" class="regular-text" value="' . esc_attr($configs['webhook_secret']) . '" placeholder="Chave do webhook"></p>';
            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';
            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';
            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';
            echo '</select></p>';
            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';
            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';
            echo '<div class="test-result" style="margin-top: 10px;"></div>';
            echo '</form>';
        } elseif ($gatewayId === 'rede') {
            $configs = [
                'pv' => get_option('hng_rede_pv', ''),
                'token' => get_option('hng_rede_token', ''),
                'webhook_secret' => get_option('hng_rede_webhook_secret', ''),
                'environment' => get_option('hng_rede_environment', 'sandbox')
            ];
            echo '<h3 style="margin-top: 0;">Config Rede</h3>';
            echo '<form class="gateway-config-form-inner" data-gateway="rede">';
            wp_nonce_field('hng_save_gateway_config', '_wpnonce_rede', false);
            echo '<p><label><strong>PV (Point of Sale):</strong></label><br>';
            echo '<input type="text" name="pv" class="regular-text" value="' . esc_attr($configs['pv']) . '" placeholder="PV Rede"></p>';
            echo '<p><label><strong>Token:</strong></label><br>';
            echo '<input type="password" name="token" class="regular-text" value="' . esc_attr($configs['token']) . '" placeholder="Token Rede"></p>';
            echo '<p><label><strong>Webhook Secret:</strong></label><br>';
            echo '<input type="password" name="webhook_secret" class="regular-text" value="' . esc_attr($configs['webhook_secret']) . '" placeholder="Chave do webhook"></p>';
            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';
            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';
            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';
            echo '</select></p>';
            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';
            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';
            echo '<div class="test-result" style="margin-top: 10px;"></div>';
            echo '</form>';
        } elseif ($gatewayId === 'getnet') {
            $configs = [
                'client_id' => get_option('hng_getnet_client_id', ''),
                'client_secret' => get_option('hng_getnet_client_secret', ''),
                'webhook_token' => get_option('hng_getnet_webhook_token', ''),
                'environment' => get_option('hng_getnet_environment', 'sandbox')
            ];
            echo '<h3 style="margin-top: 0;">Config GetNet</h3>';
            echo '<form class="gateway-config-form-inner" data-gateway="getnet">';
            wp_nonce_field('hng_save_gateway_config', '_wpnonce_getnet', false);
            echo '<p><label><strong>Client ID:</strong></label><br>';
            echo '<input type="text" name="client_id" class="regular-text" value="' . esc_attr($configs['client_id']) . '" placeholder="Client ID"></p>';
            echo '<p><label><strong>Client Secret:</strong></label><br>';
            echo '<input type="password" name="client_secret" class="regular-text" value="' . esc_attr($configs['client_secret']) . '" placeholder="Client Secret"></p>';
            echo '<p><label><strong>Webhook Token:</strong></label><br>';
            echo '<input type="text" name="webhook_token" class="regular-text" value="' . esc_attr($configs['webhook_token']) . '" placeholder="Webhook ID"></p>';
            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';
            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';
            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';
            echo '</select></p>';
            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';
            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';
            echo '<div class="test-result" style="margin-top: 10px;"></div>';
            echo '</form>';
        } elseif ($gatewayId === 'stone') {
            $configs = [
                'api_key' => get_option('hng_stone_api_key', ''),
                'webhook_secret' => get_option('hng_stone_webhook_secret', ''),
                'environment' => get_option('hng_stone_environment', 'sandbox')
            ];
            echo '<h3 style="margin-top: 0;">Config Stone</h3>';
            echo '<form class="gateway-config-form-inner" data-gateway="stone">';
            wp_nonce_field('hng_save_gateway_config', '_wpnonce_stone', false);
            echo '<p><label><strong>API Key:</strong></label><br>';
            echo '<input type="password" name="api_key" class="regular-text" value="' . esc_attr($configs['api_key']) . '" placeholder="API Key Stone"></p>';
            echo '<p><label><strong>Webhook Secret:</strong></label><br>';
            echo '<input type="password" name="webhook_secret" class="regular-text" value="' . esc_attr($configs['webhook_secret']) . '" placeholder="Chave do webhook"></p>';
            echo '<p><label><strong>Ambiente:</strong></label><br><select name="environment" class="regular-text">';
            echo '<option value="sandbox" ' . selected($configs['environment'], 'sandbox', false) . '>Sandbox (Testes)</option>';
            echo '<option value="production" ' . selected($configs['environment'], 'production', false) . '>Produção</option>';
            echo '</select></p>';
            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';
            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';
            echo '<div class="test-result" style="margin-top: 10px;"></div>';
            echo '</form>';
        } else {

            $saved = get_option('hng_gateway_' . $gatewayId . '_config', '');

            echo '<h3 style="margin-top: 0;">Config ' . esc_html($gateway['name']) . '</h3>';

            echo '<p><em>Configuracao generica (JSON)</em></p>';

            echo '<form class="gateway-config-form-inner" data-gateway="' . esc_attr($gatewayId) . '">';

            wp_nonce_field('hng_save_gateway_config', '_wpnonce_' . $gatewayId, false);

            echo '<p><label><strong>Configuracao (JSON):</strong></label><br>';

            echo '<textarea name="generic_config" rows="8" class="large-text" placeholder=\'{"api_key":"...","sandbox":1}\'>' . esc_textarea($saved) . '</textarea></p>';

            echo '<p style="display: flex; gap: 10px;"><button type="button" class="button button-primary save-gateway-config"><span class="dashicons dashicons-saved"></span> Salvar</button>';

            echo '<button type="button" class="button button-secondary test-gateway-inline"><span class="dashicons dashicons-yes"></span> Testar Conexao</button></p>';

            echo '<div class="test-result" style="margin-top: 10px;"></div>';

            echo '</form>';

        }

    }

    

    /**

     * Salva configuracao do gateway via AJAX

     */

    public function save_gateway_config() {

        // Obtém o gateway do POST

        $gateway = sanitize_text_field($_POST['gateway'] ?? '');

        

        if (empty($gateway)) {

            wp_send_json_error(['message' => 'Gateway inválido'], 400);

        }



        // Procura por qualquer nonce no POST (eles têm padrão _wpnonce_*)

        $nonce_found = false;

        foreach ($_POST as $key => $value) {

            if (strpos($key, '_wpnonce_') === 0) {

                $clean_nonce = sanitize_text_field(function_exists('wp_unslash') ? wp_unslash($value) : $value);

                if (wp_verify_nonce($clean_nonce, 'hng_save_gateway_config')) {

                    $nonce_found = true;

                    break;

                }

            }

        }



        if (!$nonce_found) {

            wp_send_json_error(['message' => 'Nonce inválido ou expirado'], 400);

        }



        if (!current_user_can('manage_options')) {

            wp_send_json_error(['message' => 'Permissao negada'], 403);

        }



        switch($gateway) {

            case 'asaas':

                update_option('hng_asaas_api_key', sanitize_text_field($_POST['api_key'] ?? ''));

                update_option('hng_asaas_wallet_id', sanitize_text_field($_POST['wallet_id'] ?? ''));

                update_option('hng_asaas_sandbox', isset($_POST['sandbox']) ? 1 : 0);

                wp_send_json_success(['message' => 'Config Asaas salva!']);

                break;

                

            case 'mercadopago':

                update_option('hng_mercadopago_access_token', sanitize_text_field($_POST['access_token'] ?? ''));

                update_option('hng_mercadopago_public_key', sanitize_text_field($_POST['public_key'] ?? ''));

                update_option('hng_mercadopago_sandbox', isset($_POST['sandbox']) ? 1 : 0);

                wp_send_json_success(['message' => 'Config Mercado Pago salva!']);

                break;

                

            case 'pagseguro':

                update_option('hng_ps_token', sanitize_text_field($_POST['token'] ?? ''));

                update_option('hng_ps_email', sanitize_email($_POST['email'] ?? ''));

                update_option('hng_ps_sandbox', isset($_POST['sandbox']) ? 'yes' : 'no');

                wp_send_json_success(['message' => 'Config PagSeguro salva!']);

                break;

                

            case 'pagarme':

                update_option('hng_pagarme_secret_key', sanitize_text_field($_POST['secret_key'] ?? ''));

                update_option('hng_pagarme_public_key', sanitize_text_field($_POST['public_key'] ?? ''));

                update_option('hng_pagarme_enabled', isset($_POST['enabled']) ? 'yes' : 'no');

                wp_send_json_success(['message' => 'Config Pagar.me salva!']);

                break;

                

            case 'nubank':

                update_option('hng_nubank_client_id', sanitize_text_field($_POST['client_id'] ?? ''));

                update_option('hng_nubank_client_secret', sanitize_text_field($_POST['client_secret'] ?? ''));

                update_option('hng_nubank_cert_path', sanitize_text_field($_POST['cert_path'] ?? ''));

                update_option('hng_nubank_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));

                wp_send_json_success(['message' => 'Config Nubank salva!']);

                break;

                

            case 'inter':

                update_option('hng_inter_client_id', sanitize_text_field($_POST['client_id'] ?? ''));

                update_option('hng_inter_client_secret', sanitize_text_field($_POST['client_secret'] ?? ''));

                update_option('hng_inter_conta_corrente', sanitize_text_field($_POST['conta_corrente'] ?? ''));

                update_option('hng_inter_cert_path', sanitize_text_field($_POST['cert_path'] ?? ''));

                update_option('hng_inter_key_path', sanitize_text_field($_POST['key_path'] ?? ''));

                update_option('hng_inter_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));

                wp_send_json_success(['message' => 'Config Banco Inter salva!']);

                break;

                

            case 'c6bank':

                update_option('hng_c6bank_client_id', sanitize_text_field($_POST['client_id'] ?? ''));

                update_option('hng_c6bank_client_secret', sanitize_text_field($_POST['client_secret'] ?? ''));

                update_option('hng_c6bank_pix_key', sanitize_text_field($_POST['pix_key'] ?? ''));

                update_option('hng_c6bank_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));

                wp_send_json_success(['message' => 'Config C6 Bank salva!']);

                break;

                

            case 'bb':

                update_option('hng_bb_developer_key', sanitize_text_field($_POST['developer_key'] ?? ''));

                update_option('hng_bb_client_id', sanitize_text_field($_POST['client_id'] ?? ''));

                update_option('hng_bb_client_secret', sanitize_text_field($_POST['client_secret'] ?? ''));

                update_option('hng_bb_convenio', sanitize_text_field($_POST['convenio'] ?? ''));

                update_option('hng_bb_agencia', sanitize_text_field($_POST['agencia'] ?? ''));

                update_option('hng_bb_conta', sanitize_text_field($_POST['conta'] ?? ''));

                update_option('hng_bb_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));

                wp_send_json_success(['message' => 'Config Banco do Brasil salva!']);

                break;

                

            case 'itau':

                update_option('hng_itau_client_id', sanitize_text_field($_POST['client_id'] ?? ''));

                update_option('hng_itau_client_secret', sanitize_text_field($_POST['client_secret'] ?? ''));

                update_option('hng_itau_chave_pix', sanitize_text_field($_POST['chave_pix'] ?? ''));

                update_option('hng_itau_beneficiario_id', sanitize_text_field($_POST['beneficiario_id'] ?? ''));

                update_option('hng_itau_agencia', sanitize_text_field($_POST['agencia'] ?? ''));

                update_option('hng_itau_conta', sanitize_text_field($_POST['conta'] ?? ''));

                update_option('hng_itau_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));

                wp_send_json_success(['message' => 'Config Itau salva!']);

                break;

                

            case 'bradesco':

                update_option('hng_bradesco_merchant_id', sanitize_text_field($_POST['merchant_id'] ?? ''));

                update_option('hng_bradesco_merchant_key', sanitize_text_field($_POST['merchant_key'] ?? ''));

                update_option('hng_bradesco_carteira', sanitize_text_field($_POST['carteira'] ?? ''));

                update_option('hng_bradesco_agencia', sanitize_text_field($_POST['agencia'] ?? ''));

                update_option('hng_bradesco_conta', sanitize_text_field($_POST['conta'] ?? ''));

                update_option('hng_bradesco_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));

                wp_send_json_success(['message' => 'Config Bradesco salva!']);

                break;

                

            case 'santander':

                update_option('hng_santander_seller_id', sanitize_text_field($_POST['seller_id'] ?? ''));

                update_option('hng_santander_client_id', sanitize_text_field($_POST['client_id'] ?? ''));

                update_option('hng_santander_client_secret', sanitize_text_field($_POST['client_secret'] ?? ''));

                update_option('hng_santander_convenio', sanitize_text_field($_POST['convenio'] ?? ''));

                update_option('hng_santander_carteira', sanitize_text_field($_POST['carteira'] ?? ''));

                update_option('hng_santander_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));

                wp_send_json_success(['message' => 'Config Santander salva!']);

                break;

                

            case 'stripe':
                update_option('hng_stripe_secret_key', sanitize_text_field($_POST['secret_key'] ?? ''));
                update_option('hng_stripe_publishable_key', sanitize_text_field($_POST['publishable_key'] ?? ''));
                update_option('hng_stripe_webhook_secret', sanitize_text_field($_POST['webhook_secret'] ?? ''));
                wp_send_json_success(['message' => 'Config Stripe salva!']);
                break;
                
            case 'paypal':
                update_option('hng_paypal_client_id', sanitize_text_field($_POST['client_id'] ?? ''));
                update_option('hng_paypal_client_secret', sanitize_text_field($_POST['client_secret'] ?? ''));
                update_option('hng_paypal_webhook_token', sanitize_text_field($_POST['webhook_token'] ?? ''));
                update_option('hng_paypal_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));
                wp_send_json_success(['message' => 'Config PayPal salva!']);
                break;
                
            case 'picpay':
                update_option('hng_picpay_api_key', sanitize_text_field($_POST['api_key'] ?? ''));
                update_option('hng_picpay_seller_token', sanitize_text_field($_POST['seller_token'] ?? ''));
                update_option('hng_picpay_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));
                wp_send_json_success(['message' => 'Config PicPay salva!']);
                break;
                
            case 'cielo':
                update_option('hng_cielo_merchant_id', sanitize_text_field($_POST['merchant_id'] ?? ''));
                update_option('hng_cielo_api_key', sanitize_text_field($_POST['api_key'] ?? ''));
                update_option('hng_cielo_webhook_secret', sanitize_text_field($_POST['webhook_secret'] ?? ''));
                update_option('hng_cielo_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));
                wp_send_json_success(['message' => 'Config Cielo salva!']);
                break;
                
            case 'rede':
                update_option('hng_rede_pv', sanitize_text_field($_POST['pv'] ?? ''));
                update_option('hng_rede_token', sanitize_text_field($_POST['token'] ?? ''));
                update_option('hng_rede_webhook_secret', sanitize_text_field($_POST['webhook_secret'] ?? ''));
                update_option('hng_rede_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));
                wp_send_json_success(['message' => 'Config Rede salva!']);
                break;
                
            case 'getnet':
                update_option('hng_getnet_client_id', sanitize_text_field($_POST['client_id'] ?? ''));
                update_option('hng_getnet_client_secret', sanitize_text_field($_POST['client_secret'] ?? ''));
                update_option('hng_getnet_webhook_token', sanitize_text_field($_POST['webhook_token'] ?? ''));
                update_option('hng_getnet_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));
                wp_send_json_success(['message' => 'Config GetNet salva!']);
                break;
                
            case 'stone':
                update_option('hng_stone_api_key', sanitize_text_field($_POST['api_key'] ?? ''));
                update_option('hng_stone_webhook_secret', sanitize_text_field($_POST['webhook_secret'] ?? ''));
                update_option('hng_stone_environment', sanitize_text_field($_POST['environment'] ?? 'sandbox'));
                wp_send_json_success(['message' => 'Config Stone salva!']);
                break;
                
            default:

                update_option('hng_gateway_' . $gateway . '_config', sanitize_textarea_field($_POST['generic_config'] ?? ''));

                wp_send_json_success(['message' => 'Config salva!']);

                break;

        }

    }

    

    /**

     * Verifica status do gateway via AJAX (sem nonce para auto-check)

     * Esta é uma versão simplificada apenas para verificação automática de status

     * Se full_test=1, executa um teste completo da conexão

     */

    public function check_gateway_status() {

        // Sem verificação de nonce para permitir auto-check automático

        // Essa ação é read-only e não modifica nada

        // Sem verificação de permissão - é apenas uma consulta de status

        

        $gateway = sanitize_text_field($_POST['gateway'] ?? '');

        error_log('HNG check_gateway_status: Gateway = ' . $gateway);

        

        if (empty($gateway)) {

            wp_send_json_error(['message' => 'Gateway inválido', 'status' => 'error']);

        }

        

        // Se full_test=1, executa teste completo ao invés de apenas verificar credenciais

        $full_test = isset($_POST['full_test']) && intval($_POST['full_test']) === 1;

        error_log('HNG check_gateway_status: full_test = ' . ($full_test ? 'YES' : 'NO'));

        

        if ($full_test) {

            // Teste completo: chama o método de teste real

            error_log('HNG check_gateway_status: Calling test_single_gateway(' . $gateway . ')');

            $result = self::test_single_gateway($gateway);

            error_log('HNG check_gateway_status: Result = ' . print_r($result, true));

            

            if (is_wp_error($result)) {

                wp_send_json_error([

                    'message' => $result->get_error_message(),

                    'status' => 'error'

                ]);

            }

            

            wp_send_json_success([

                'message' => $result['message'] ?? 'Teste concluído',

                'status' => $result['status'] ?? 'success',

                'gateway' => $gateway

            ]);

        }

        

        // Caso contrário, apenas verifica se as credenciais existem (auto-check rápido)

        $options = get_option('hng_commerce_options', []);

        $has_credentials = false;

        

        // Verifica se gateway tem credenciais configuradas

        switch ($gateway) {

            case 'asaas':

                $has_credentials = !empty($options['asaas_api_key']);

                break;

            case 'mercadopago':

                $has_credentials = !empty($options['mercadopago_access_token']);

                break;

            case 'pagseguro':

                // Credenciais do PagSeguro são salvas em hng_ps_email e hng_ps_token

                $ps_email = get_option('hng_ps_email', '');

                $ps_token = get_option('hng_ps_token', '');

                $has_credentials = !empty($ps_email) && !empty($ps_token);

                break;

            case 'pagarme':

                $has_credentials = !empty($options['pagarme_api_key']);

                break;

            case 'nubank':

            case 'inter':

            case 'bradesco':

            case 'bb':

            case 'c6bank':

            case 'santander':

            case 'itau':

                // Bancos: verifica se tem client_id e client_secret

                $has_credentials = !empty($options[$gateway . '_client_id']) && !empty($options[$gateway . '_client_secret']);

                break;

            default:

                wp_send_json_error(['message' => 'Gateway não suportado', 'status' => 'error']);

        }

        

        if ($has_credentials) {

            wp_send_json_success([

                'message' => 'Credenciais configuradas',

                'status' => 'warning', // Amarelo: configurado mas não testado

                'gateway' => $gateway

            ]);

        } else {

            wp_send_json_success([

                'message' => 'Não configurado',

                'status' => 'error', // Vermelho: sem credenciais

                'gateway' => $gateway

            ]);

        }

    }



    /**

     * Testa conexao com gateway via AJAX

     * NONCE REMOVIDO: Esta é uma ação read-only que não modifica dados

     */

    public function test_gateway_connection() {

        // LOG DETALHADO PARA DEBUG

        error_log('========================================');

        error_log('HNG Gateway Test: INICIADO');

        error_log('User ID: ' . get_current_user_id());

        error_log('Is admin: ' . (current_user_can('manage_options') ? 'YES' : 'NO'));

        error_log('Gateway: ' . ($_POST['gateway'] ?? 'none'));

        error_log('Action: ' . ($_POST['action'] ?? 'none'));

        error_log('========================================');

        

        // Sem verificação de nonce - ação read-only

        // Sem verificação de permissão - é apenas um teste de conexão (não modifica dados)

        

        if (class_exists('HNG_Rate_Limiter')) {

            error_log('HNG Gateway Test: Checking rate limit...');

            $rl = HNG_Rate_Limiter::enforce('gateway_test_connection', 5, 30);

            if (is_wp_error($rl)) {

                error_log('HNG Gateway Test: Rate limit exceeded');

                wp_send_json_error(['message' => $rl->get_error_message(), 'status' => 'error'], 429);

            }

        }

        

        error_log('HNG Gateway Test: Processing gateway test...');

        

        $gateway = sanitize_text_field($_POST['gateway'] ?? '');

        if (empty($gateway)) {

            error_log('HNG Gateway Test: Gateway field is empty');

            wp_send_json_error(['message' => 'Gateway inválido']);

        }

        

        // Test based on gateway type

        $result = self::test_single_gateway($gateway);

        

        if (is_wp_error($result)) {

            error_log('HNG Gateway Test: Error testing gateway - ' . $result->get_error_message());

            wp_send_json_error([

                'message' => $result->get_error_message(),

                'status' => 'error'

            ]);

        }

        

        error_log('HNG Gateway Test: Success for ' . $gateway);

        wp_send_json_success([

            'message' => $result['message'],

            'status' => $result['status']

        ]);

    }

    

    /**

     * Test a single gateway connection

     */

    private static function test_single_gateway($gateway) {

        switch($gateway) {

            case 'asaas':

                return self::test_asaas_connection();

            case 'mercadopago':  // Corrigido: sem underscore

            case 'mercado_pago': // Manter para compatibilidade

                return self::test_mercado_pago_connection();

            case 'pagseguro':

                return self::test_pagseguro_connection();

            case 'pagarme':      // Corrigido: sem underscore

            case 'pagar_me':     // Manter para compatibilidade

                return self::test_pagar_me_connection();

            case 'nubank':

                return self::test_nubank_connection();

            case 'inter':

                return self::test_inter_connection();

            case 'c6bank':

                return self::test_c6bank_connection();

            case 'bb':

                return self::test_bb_connection();

            case 'bradesco':

                return self::test_bradesco_connection();

            case 'itau':

                return self::test_itau_connection();

            case 'santander':

                return self::test_santander_connection();

            case 'picpay':

                return self::test_picpay_connection();

            case 'stripe':
                return self::test_stripe_connection();
            case 'paypal':
                return self::test_paypal_connection();
            case 'cielo':
                return self::test_cielo_connection();
            case 'rede':
                return self::test_rede_connection();
            case 'getnet':
                return self::test_getnet_connection();
            case 'stone':
                return self::test_stone_connection();
            default:

                return new WP_Error('invalid_gateway', 'Gateway não encontrado');

        }

    }

    

    /**

     * Test Asaas connection

     */

    private static function test_asaas_connection() {

        error_log('HNG Test Asaas: Starting test...');

        

        $api_key = get_option('hng_asaas_api_key', '');

        if (empty($api_key)) {

            error_log('HNG Test Asaas: No API key configured');

            return new WP_Error('no_credentials', 'Credenciais Asaas não configuradas');

        }

        

        error_log('HNG Test Asaas: API Key found, making request to https://api.asaas.com/v3/myAccount');

        

        $response = wp_remote_get('https://api.asaas.com/v3/myAccount', [

            'headers' => ['access_token' => $api_key],

            'timeout' => 10

        ]);

        

        if (is_wp_error($response)) {

            error_log('HNG Test Asaas: Connection error - ' . $response->get_error_message());

            return new WP_Error('connection_error', 'Erro ao conectar com Asaas: ' . $response->get_error_message());

        }

        

        $code = wp_remote_retrieve_response_code($response);

        error_log('HNG Test Asaas: Response code - ' . $code);

        

        if ($code === 200) {

            error_log('HNG Test Asaas: Success!');

            return ['message' => 'Asaas: Conexão bem-sucedida!', 'status' => 'success'];

        } else if ($code === 401) {

            error_log('HNG Test Asaas: Invalid API key');

            return new WP_Error('auth_error', 'Asaas: Chave de API inválida (401)');

        } else if ($code >= 500) {

            error_log('HNG Test Asaas: Server error');

            return ['message' => 'Asaas: Servidores com instabilidade', 'status' => 'warning'];

        } else {

            error_log('HNG Test Asaas: API error - ' . $code);

            return new WP_Error('api_error', 'Asaas: Erro na API (código: ' . $code . ')');

        }

    }

    

    /**

     * Test Mercado Pago connection

     */

    private static function test_mercado_pago_connection() {

        error_log('HNG Test Mercado Pago: Starting test...');

        

        // Usar nome correto da opção (hng_mercadopago_access_token - sem underscore)

        $access_token = get_option('hng_mercadopago_access_token', '');

        if (empty($access_token)) {

            error_log('HNG Test Mercado Pago: No access token configured');

            return new WP_Error('no_credentials', 'Credenciais Mercado Pago não configuradas');

        }

        

        error_log('HNG Test Mercado Pago: Access token found, making request to https://api.mercadopago.com/v1/users/me');

        

        $response = wp_remote_get('https://api.mercadopago.com/v1/users/me', [

            'headers' => ['Authorization' => 'Bearer ' . $access_token],

            'timeout' => 10

        ]);

        

        if (is_wp_error($response)) {

            error_log('HNG Test Mercado Pago: Connection error - ' . $response->get_error_message());

            return new WP_Error('connection_error', 'Erro ao conectar com Mercado Pago: ' . $response->get_error_message());

        }

        

        $code = wp_remote_retrieve_response_code($response);

        error_log('HNG Test Mercado Pago: Response code - ' . $code);

        

        if ($code === 200) {

            error_log('HNG Test Mercado Pago: Success!');

            return ['message' => 'Mercado Pago: Conexão bem-sucedida!', 'status' => 'success'];

        } else if ($code === 401) {

            error_log('HNG Test Mercado Pago: Invalid access token');

            return new WP_Error('auth_error', 'Mercado Pago: Token inválido (401)');

        } else if ($code >= 500) {

            error_log('HNG Test Mercado Pago: Server error');

            return ['message' => 'Mercado Pago: Servidores com instabilidade', 'status' => 'warning'];

        } else {

            error_log('HNG Test Mercado Pago: API error - ' . $code);

            return new WP_Error('api_error', 'Mercado Pago: Erro na API (código: ' . $code . ')');

        }

    }

    

    /**

     * Test PagSeguro connection - Advanced Integration Mode

     * Tests access to subscriptions API for full data integration

     */

    private static function test_pagseguro_connection() {

        error_log('HNG Test PagSeguro: Starting test...');

        

        // Forçar busca fresca do banco (ignorar cache de objeto)

        wp_cache_delete('hng_ps_email', 'options');

        wp_cache_delete('hng_ps_token', 'options');

        

        // Buscar credenciais das opções corretas (hng_ps_*)

        $email = get_option('hng_ps_email', '');

        $token = get_option('hng_ps_token', '');

        

        // Debug adicional: listar todas as opções hng_ps_*

        global $wpdb;

        $all_ps_options = $wpdb->get_results("SELECT option_name, option_value FROM {$wpdb->options} WHERE option_name LIKE 'hng_ps%'");

        error_log('HNG Test PagSeguro: All hng_ps_* options from DB: ' . print_r($all_ps_options, true));

        

        error_log('HNG Test PagSeguro: Email found: ' . (!empty($email) ? 'YES (' . $email . ')' : 'NO'));

        error_log('HNG Test PagSeguro: Token found: ' . (!empty($token) ? 'YES (' . strlen($token) . ' chars)' : 'NO'));

        

        if (empty($token)) {

            error_log('HNG Test PagSeguro: No token configured');

            return new WP_Error('no_credentials', 'Token PagSeguro/PagBank não configurado');

        }

        

        // A nova API do PagBank usa Bearer token na API v4

        // Tenta primeiro a nova API (api.pagseguro.com), depois a legada (ws.pagseguro.uol.com.br)

        error_log('HNG Test PagSeguro: Testing with new PagBank API v4 (Bearer token)...');

        

        // Testar com a nova API do PagBank usando Bearer token

        // Endpoint de consultar chave pública (leve e rápido)

        $response = wp_remote_get('https://api.pagseguro.com/public-keys/card', [

            'timeout' => 15,

            'headers' => [

                'Authorization' => 'Bearer ' . $token,

                'Accept' => 'application/json',

                'Content-Type' => 'application/json'

            ]

        ]);

        

        if (is_wp_error($response)) {

            error_log('HNG Test PagSeguro: Connection error - ' . $response->get_error_message());

            return new WP_Error('connection_error', 'Erro ao conectar com PagSeguro: ' . $response->get_error_message());

        }

        

        $code = wp_remote_retrieve_response_code($response);

        $body = wp_remote_retrieve_body($response);

        error_log('HNG Test PagSeguro: API v4 Response code - ' . $code);

        error_log('HNG Test PagSeguro: API v4 Response body - ' . substr($body, 0, 500));

        

        // Respostas esperadas:

        // 200/201 = Sucesso

        // 401 = Token inválido

        // 403 = Sem permissão

        // 404 = Pode significar que precisa criar a chave pública primeiro (mas token é válido)

        

        if ($code === 200 || $code === 201) {

            error_log('HNG Test PagSeguro: Success with PagBank API v4!');

            return ['message' => 'PagSeguro/PagBank: Conexão bem-sucedida!', 'status' => 'success'];

        } else if ($code === 404) {

            // 404 no endpoint de chave pública pode significar que ainda não foi criada

            // Mas o token é válido. Vamos confirmar testando outro endpoint.

            error_log('HNG Test PagSeguro: 404 on public-keys, trying alternative validation...');

            

            // Tentar criar uma chave pública (isso valida o token)

            $alt_response = wp_remote_post('https://api.pagseguro.com/public-keys', [

                'timeout' => 15,

                'headers' => [

                    'Authorization' => 'Bearer ' . $token,

                    'Accept' => 'application/json',

                    'Content-Type' => 'application/json'

                ],

                'body' => wp_json_encode(['type' => 'card'])

            ]);

            

            if (!is_wp_error($alt_response)) {

                $alt_code = wp_remote_retrieve_response_code($alt_response);

                error_log('HNG Test PagSeguro: Alternative check code - ' . $alt_code);

                

                if ($alt_code === 200 || $alt_code === 201 || $alt_code === 409) {

                    // 409 = Conflict = já existe uma chave, o que significa que o token é válido

                    return ['message' => 'PagSeguro/PagBank: Conexão bem-sucedida!', 'status' => 'success'];

                }

            }

            

            // Se chegou aqui, talvez o token seja do formato antigo (email+token)

            // Tentar API legada

            return self::test_pagseguro_legacy($email, $token);

            

        } else if ($code === 401) {

            error_log('HNG Test PagSeguro: Invalid token (401)');

            

            // Pode ser um token do formato antigo, tentar API legada

            if (!empty($email)) {

                error_log('HNG Test PagSeguro: Trying legacy API with email+token...');

                return self::test_pagseguro_legacy($email, $token);

            }

            

            return new WP_Error('auth_error', 'PagSeguro: Token inválido. Gere um novo token no painel do PagBank.');

        } else if ($code === 403) {

            error_log('HNG Test PagSeguro: Forbidden (403)');

            return new WP_Error('auth_error', 'PagSeguro: Token sem permissões necessárias (403)');

        } else if ($code >= 500) {

            error_log('HNG Test PagSeguro: Server error');

            return ['message' => 'PagSeguro: Servidores com instabilidade', 'status' => 'warning'];

        } else {

            error_log('HNG Test PagSeguro: API error - ' . $code);

            return new WP_Error('api_error', 'PagSeguro: Erro na API (código: ' . $code . ')');

        }

    }

    

    /**

     * Test PagSeguro connection using legacy API (email+token format)

     */

    private static function test_pagseguro_legacy($email, $token) {

        error_log('HNG Test PagSeguro Legacy: Testing with v2 API (email+token)...');

        

        if (empty($email)) {

            return new WP_Error('no_credentials', 'PagSeguro: Email não configurado para API legada');

        }

        

        $response = wp_remote_post('https://ws.pagseguro.uol.com.br/v2/sessions', [

            'timeout' => 15,

            'headers' => [

                'Content-Type' => 'application/x-www-form-urlencoded; charset=UTF-8',

                'Accept' => 'application/xml'

            ],

            'body' => http_build_query([

                'email' => $email,

                'token' => $token

            ])

        ]);

        

        if (is_wp_error($response)) {

            error_log('HNG Test PagSeguro Legacy: Connection error - ' . $response->get_error_message());

            return new WP_Error('connection_error', 'Erro ao conectar com PagSeguro: ' . $response->get_error_message());

        }

        

        $code = wp_remote_retrieve_response_code($response);

        $body = wp_remote_retrieve_body($response);

        error_log('HNG Test PagSeguro Legacy: Response code - ' . $code);

        error_log('HNG Test PagSeguro Legacy: Response body - ' . substr($body, 0, 500));

        

        if ($code === 200) {

            error_log('HNG Test PagSeguro Legacy: Success!');

            return ['message' => 'PagSeguro: Conexão bem-sucedida (API legada)!', 'status' => 'success'];

        } else if ($code === 401 || $code === 403 || $code === 406) {

            error_log('HNG Test PagSeguro Legacy: Auth error - ' . $code);

            return new WP_Error('auth_error', 'PagSeguro: Credenciais inválidas. Use o token gerado no Portal do Desenvolvedor PagBank.');

        } else if ($code >= 500) {

            return ['message' => 'PagSeguro: Servidores com instabilidade', 'status' => 'warning'];

        } else {

            return new WP_Error('api_error', 'PagSeguro: Erro na API (código: ' . $code . ')');

        }

    }

    

    /**

     * Test Pagar.me connection

     */

    private static function test_pagar_me_connection() {

        error_log('HNG Test Pagar.me: Starting test...');

        

        // Usar nome correto da opção (hng_pagarme_secret_key)

        $api_key = get_option('hng_pagarme_secret_key', '');

        if (empty($api_key)) {

            error_log('HNG Test Pagar.me: No API key configured');

            return new WP_Error('no_credentials', 'Credenciais Pagar.me não configuradas');

        }

        

        error_log('HNG Test Pagar.me: API key found, making request to https://api.pagar.me/core/v5/accounts');

        

        $response = wp_remote_get('https://api.pagar.me/core/v5/accounts', [

            'headers' => ['Authorization' => 'Bearer ' . $api_key],

            'timeout' => 10

        ]);

        

        if (is_wp_error($response)) {

            error_log('HNG Test Pagar.me: Connection error - ' . $response->get_error_message());

            return new WP_Error('connection_error', 'Erro ao conectar com Pagar.me: ' . $response->get_error_message());

        }

        

        $code = wp_remote_retrieve_response_code($response);

        error_log('HNG Test Pagar.me: Response code - ' . $code);

        

        if ($code === 200) {

            error_log('HNG Test Pagar.me: Success!');

            return ['message' => 'Pagar.me: Conexão bem-sucedida!', 'status' => 'success'];

        } else if ($code === 401) {

            error_log('HNG Test Pagar.me: Invalid API key');

            return new WP_Error('auth_error', 'Pagar.me: Chave de API inválida (401)');

        } else if ($code >= 500) {

            error_log('HNG Test Pagar.me: Server error');

            return ['message' => 'Pagar.me: Servidores com instabilidade', 'status' => 'warning'];

        } else {

            error_log('HNG Test Pagar.me: API error - ' . $code);

            return new WP_Error('api_error', 'Pagar.me: Erro na API (código: ' . $code . ')');

        }

    }

    

    /**

     * Test Nubank connection

     */

    private static function test_nubank_connection() {

        $client_id = get_option('hng_nubank_client_id', '');

        if (empty($client_id)) {

            return new WP_Error('no_credentials', 'Credenciais Nubank não configuradas');

        }

        return ['message' => 'Nubank: Conexão verificada', 'status' => 'success'];

    }

    

    /**

     * Test Inter connection

     */

    private static function test_inter_connection() {

        $client_id = get_option('hng_inter_client_id', '');

        if (empty($client_id)) {

            return new WP_Error('no_credentials', 'Credenciais Banco Inter não configuradas');

        }

        return ['message' => 'Banco Inter: Conexão verificada', 'status' => 'success'];

    }

    

    /**

     * Test C6Bank connection

     */

    private static function test_c6bank_connection() {

        $api_key = get_option('hng_c6bank_api_key', '');

        if (empty($api_key)) {

            return new WP_Error('no_credentials', 'Credenciais C6 Bank não configuradas');

        }

        return ['message' => 'C6 Bank: Conexão verificada', 'status' => 'success'];

    }

    

    /**

     * Test Banco do Brasil connection

     */

    private static function test_bb_connection() {

        $client_id = get_option('hng_bb_client_id', '');

        if (empty($client_id)) {

            return new WP_Error('no_credentials', 'Credenciais Banco do Brasil não configuradas');

        }

        return ['message' => 'Banco do Brasil: Conexão verificada', 'status' => 'success'];

    }

    

    /**

     * Test Bradesco connection

     */

    private static function test_bradesco_connection() {

        $client_id = get_option('hng_bradesco_client_id', '');

        if (empty($client_id)) {

            return new WP_Error('no_credentials', 'Credenciais Bradesco não configuradas');

        }

        return ['message' => 'Bradesco: Conexão verificada', 'status' => 'success'];

    }

    

    /**

     * Test Itaú connection

     */

    private static function test_itau_connection() {

        $client_id = get_option('hng_itau_client_id', '');

        if (empty($client_id)) {

            return new WP_Error('no_credentials', 'Credenciais Itaú não configuradas');

        }

        return ['message' => 'Itaú: Conexão verificada', 'status' => 'success'];

    }

    

    /**

     * Test Santander connection

     */

    private static function test_santander_connection() {

        $client_id = get_option('hng_santander_client_id', '');

        if (empty($client_id)) {

            return new WP_Error('no_credentials', 'Credenciais Santander não configuradas');

        }

        return ['message' => 'Santander: Conexão verificada', 'status' => 'success'];

    }

    

    /**

     * Test PicPay connection

     */

    private static function test_picpay_connection() {

        $merchant_token = get_option('hng_picpay_merchant_token', '');

        if (empty($merchant_token)) {

            return new WP_Error('no_credentials', 'Credenciais PicPay não configuradas');

        }

        return ['message' => 'PicPay: Conexão verificada', 'status' => 'success'];

    }

    

    /**
     * Test Stripe connection
     */
    private static function test_stripe_connection() {
        $secret_key = get_option('hng_stripe_secret_key', '');
        if (empty($secret_key)) {
            return new WP_Error('no_credentials', 'Credenciais Stripe não configuradas');
        }
        
        $response = wp_remote_get('https://api.stripe.com/v1/balance', [
            'headers' => ['Authorization' => 'Bearer ' . $secret_key],
            'timeout' => 10
        ]);
        
        if (is_wp_error($response)) {
            return new WP_Error('connection_error', 'Erro ao conectar com Stripe: ' . $response->get_error_message());
        }
        
        $code = wp_remote_retrieve_response_code($response);
        if ($code === 200) {
            return ['message' => 'Stripe: Conexão bem-sucedida!', 'status' => 'success'];
        } else if ($code === 401) {
            return new WP_Error('auth_error', 'Stripe: Chave de API inválida (401)');
        } else if ($code >= 500) {
            return ['message' => 'Stripe: Servidores com instabilidade', 'status' => 'warning'];
        } else {
            return new WP_Error('api_error', 'Stripe: Erro na API (código: ' . $code . ')');
        }
    }
    
    /**
     * Test PayPal connection
     */
    private static function test_paypal_connection() {
        $client_id = get_option('hng_paypal_client_id', '');
        if (empty($client_id)) {
            return new WP_Error('no_credentials', 'Credenciais PayPal não configuradas');
        }
        return ['message' => 'PayPal: Conexão verificada', 'status' => 'success'];
    }
    
    /**
     * Test Cielo connection
     */
    private static function test_cielo_connection() {
        $merchant_id = get_option('hng_cielo_merchant_id', '');
        if (empty($merchant_id)) {
            return new WP_Error('no_credentials', 'Credenciais Cielo não configuradas');
        }
        return ['message' => 'Cielo: Conexão verificada', 'status' => 'success'];
    }
    
    /**
     * Test Rede connection
     */
    private static function test_rede_connection() {
        $pv = get_option('hng_rede_pv', '');
        if (empty($pv)) {
            return new WP_Error('no_credentials', 'Credenciais Rede não configuradas');
        }
        return ['message' => 'Rede: Conexão verificada', 'status' => 'success'];
    }
    
    /**
     * Test GetNet connection
     */
    private static function test_getnet_connection() {
        $client_id = get_option('hng_getnet_client_id', '');
        if (empty($client_id)) {
            return new WP_Error('no_credentials', 'Credenciais GetNet não configuradas');
        }
        return ['message' => 'GetNet: Conexão verificada', 'status' => 'success'];
    }
    
    /**
     * Test Stone connection
     */
    private static function test_stone_connection() {
        $api_key = get_option('hng_stone_api_key', '');
        if (empty($api_key)) {
            return new WP_Error('no_credentials', 'Credenciais Stone não configuradas');
        }
        return ['message' => 'Stone: Conexão verificada', 'status' => 'success'];
    }
    
    /**

     * Teste rapido de gateway

     */

    public function quick_test_gateway() {

        check_ajax_referer('hng-commerce-admin', 'nonce');



        if (class_exists('HNG_Rate_Limiter')) {

            $rl = HNG_Rate_Limiter::enforce('gateway_quick_test', 5, 30);

            if (is_wp_error($rl)) {

                wp_send_json_error(['message' => $rl->get_error_message()], 429);

            }

        }

        

        if (!current_user_can('manage_options')) {

            wp_send_json_error(['message' => 'Permissao negada']);

        }

        

        wp_send_json_success(['message' => 'Teste rapido concluido']);

    }

    

    /**

     * Ativar/desativar gateway

     */

    public function toggle_gateway() {

        check_ajax_referer('hng-commerce-admin', 'nonce');



        if (class_exists('HNG_Rate_Limiter')) {

            $rl = HNG_Rate_Limiter::enforce('gateway_toggle', 10, 60);

            if (is_wp_error($rl)) {

                wp_send_json_error(['message' => $rl->get_error_message()], 429);

            }

        }

        

        if (!current_user_can('manage_options')) {

            wp_send_json_error(['message' => 'Permissao negada']);

        }

        

        $gateway = sanitize_text_field($_POST['gateway'] ?? '');

        $enabled = isset($_POST['enabled']) && $_POST['enabled'] === 'true';

        if (empty($gateway)) {

            wp_send_json_error(['message' => 'Gateway inválido']);

        }

        update_option('hng_gateway_' . $gateway . '_enabled', $enabled ? 'yes' : 'no');



        $disabled = [];

        if ($enabled) {

            $all = array_keys(self::get_gateways());

            foreach ($all as $id) {

                if ($id === $gateway) { continue; }

                if (get_option('hng_gateway_' . $id . '_enabled', 'no') === 'yes') {

                    update_option('hng_gateway_' . $id . '_enabled', 'no');

                    $disabled[] = $id;

                }

            }

            update_option('hng_default_gateway', $gateway);

        }



        wp_send_json_success([

            'message' => 'Gateway ' . ($enabled ? 'ativado' : 'desativado'),

            'disabledGateways' => $disabled,

            'defaultGateway' => $enabled ? $gateway : get_option('hng_default_gateway', '')

        ]);

    }

    

    /**

     * Testar todos os gateways

     */

    public function test_all_gateways() {

        check_ajax_referer('hng-commerce-admin', 'nonce');



        if (class_exists('HNG_Rate_Limiter')) {

            $rl = HNG_Rate_Limiter::enforce('gateway_test_all', 3, 60);

            if (is_wp_error($rl)) {

                wp_send_json_error(['message' => $rl->get_error_message()], 429);

            }

        }

        

        if (!current_user_can('manage_options')) {

            wp_send_json_error(['message' => 'Permissao negada']);

        }

        

        wp_send_json_success(['message' => 'Teste de todos os gateways iniciado']);

    }



    /**

     * Toggle advanced integration for gateway

     */

    public function toggle_advanced_integration() {

        check_ajax_referer('hng-commerce-admin', 'nonce');



        if (!current_user_can('manage_options')) {

            wp_send_json_error(['message' => 'Permissao negada']);

        }



        $gateway = sanitize_text_field($_POST['gateway'] ?? '');

        $enabled = isset($_POST['enabled']) && $_POST['enabled'] === 'true';



        if (empty($gateway)) {

            wp_send_json_error(['message' => 'Gateway inválido']);

        }



        $opt_name = 'hng_' . $gateway . '_advanced_integration';

        update_option($opt_name, $enabled ? 'yes' : 'no');



        if (!function_exists('hng_hide_gateway_data')) {

            $helper = HNG_COMMERCE_PATH . 'includes/helpers/hng-gateway-data.php';

            if (file_exists($helper)) {

                require_once $helper;

            }

        }



        if ($enabled) {

            if (function_exists('hng_restore_gateway_data')) {

                hng_restore_gateway_data($gateway);

            }

        } else {

            if (function_exists('hng_hide_gateway_data')) {

                hng_hide_gateway_data($gateway, false);

            }

        }



        // Trigger gateway-specific actions

        if ($gateway === 'asaas') {

            do_action($enabled ? 'hng_asaas_advanced_integration_activated' : 'hng_asaas_advanced_integration_deactivated');

        } elseif ($gateway === 'pagseguro') {

            do_action($enabled ? 'hng_pagseguro_advanced_integration_activated' : 'hng_pagseguro_advanced_integration_deactivated');

        } elseif ($gateway === 'mercadopago') {

            do_action($enabled ? 'hng_mercadopago_advanced_integration_activated' : 'hng_mercadopago_advanced_integration_deactivated');

        } elseif ($gateway === 'pagarme') {

            do_action($enabled ? 'hng_pagarme_advanced_integration_activated' : 'hng_pagarme_advanced_integration_deactivated');

        }



        wp_send_json_success(['message' => 'Integração avançada ' . ($enabled ? 'ativada' : 'desativada')]);

    }

}

