<?php
/**
 * Orders Page - Gerenciamento de Pedidos
 * 
 * Página dedicada para gerenciar pedidos do sistema HNG Commerce.
 * Exibe lista de pedidos com filtros, paginação e visualização detalhada.
 * 
 * @package HNG_Commerce
 * @since 1.1.1
 */

if (!defined('ABSPATH')) {
    exit;
}

class HNG_Orders_Page {
    
    /**
     * Render orders page
     */
    public static function render() {
        // Ensure orders list table is loaded
        if (!class_exists('HNG_Orders_List')) {
            if (file_exists(HNG_COMMERCE_PATH . 'includes/admin/class-hng-orders-list.php')) {
                require_once HNG_COMMERCE_PATH . 'includes/admin/class-hng-orders-list.php';
            } else {
                echo '<div class="notice notice-error"><p>' . esc_html__('Classe HNG_Orders_List não encontrada.', 'hng-commerce') . '</p></div>';
                return;
            }
        }
        
        // Handle actions
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Action parameter verified below with nonce
        $action = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : '';
        
        if ($action === 'view') {
            self::render_order_details();
            return;
        }
        
        if ($action === 'bulk_action' && isset($_POST['hng_orders_nonce'])) {
            self::handle_bulk_actions();
        }
        
        // Render list view
        self::render_list_view();
    }
    
    /**
     * Render list view
     */
    private static function render_list_view() {
        $orders_table = new HNG_Orders_List();
        $orders_table->prepare_items();
        
        ?>
        <div class="wrap hng-admin-wrap">
            <h1 class="hng-page-title">
                <span class="dashicons dashicons-cart"></span>
                <?php esc_html_e('Pedidos', 'hng-commerce'); ?>
            </h1>
            
            <div class="hng-card" style="margin-top: 20px;">
                <div class="hng-card-content">
                    <form method="post">
                        <?php
                        $orders_table->views();
                        $orders_table->search_box(__('Buscar pedidos', 'hng-commerce'), 'orders');
                        $orders_table->display();
                        ?>
                    </form>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render order details view
     */
    private static function render_order_details() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only GET parameter, order data display only
        if (!isset($_GET['order_id'])) {
            echo '<div class="notice notice-error"><p>' . esc_html__('ID do pedido não fornecido.', 'hng-commerce') . '</p></div>';
            return;
        }
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only GET parameter for order display
        $order_id = absint(wp_unslash($_GET['order_id']));
        
        if ($order_id <= 0) {
            echo '<div class="notice notice-error"><p>' . esc_html__('ID do pedido inválido.', 'hng-commerce') . '</p></div>';
            return;
        }
        
        global $wpdb;
        
        // Get order
        $orders_table = hng_db_full_table_name('hng_orders');
        $orders_table_sql = '`' . str_replace('`', '', $orders_table) . '`';
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Order details query with sanitized table name
        // phpcs:ignore PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name sanitized via hng_db_full_table_name and backtick escaping
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$orders_table_sql} WHERE id = %d",
            $order_id
        ));
        
        if (!$order) {
            echo '<div class="notice notice-error"><p>' . esc_html__('Pedido não encontrado.', 'hng-commerce') . '</p></div>';
            return;
        }
        
        // Get order items
        $items_table = hng_db_full_table_name('hng_order_items');
        $items_table_sql = '`' . str_replace('`', '', $items_table) . '`';
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Order items query with sanitized table name
        // phpcs:ignore PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name sanitized via hng_db_full_table_name and backtick escaping
        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$items_table_sql} WHERE order_id = %d",
            $order_id
        ));
        
        ?>
        <div class="wrap hng-admin-wrap">
            <h1 class="hng-page-title">
                <span class="dashicons dashicons-cart"></span>
                <?php
                /* translators: %d: Order ID */
                echo esc_html(sprintf(__('Pedido #%d', 'hng-commerce'), $order_id));
                ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=hng-orders')); ?>" class="page-title-action">
                    <?php esc_html_e('← Voltar', 'hng-commerce'); ?>
                </a>
            </h1>
            
            <div class="hng-grid hng-grid-2" style="margin-top: 20px;">
                <!-- Order Details -->
                <div class="hng-card">
                    <div class="hng-card-header">
                        <h2 class="hng-card-title"><?php esc_html_e('Detalhes do Pedido', 'hng-commerce'); ?></h2>
                    </div>
                    <div class="hng-card-content">
                        <table class="widefat striped">
                            <tbody>
                                <tr>
                                    <th><?php esc_html_e('Status:', 'hng-commerce'); ?></th>
                                    <td>
                                        <?php
                                        $status_labels = array(
                                            'pending' => __('Pendente', 'hng-commerce'),
                                            'processing' => __('Processando', 'hng-commerce'),
                                            'completed' => __('Concluído', 'hng-commerce'),
                                            'cancelled' => __('Cancelado', 'hng-commerce'),
                                            'refunded' => __('Reembolsado', 'hng-commerce'),
                                        );
                                        $status_badge = isset($status_labels[$order->status]) ? $status_labels[$order->status] : ucfirst($order->status);
                                        echo '<span class="hng-badge hng-badge-' . esc_attr($order->status) . '">' . esc_html($status_badge) . '</span>';
                                        ?>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Data:', 'hng-commerce'); ?></th>
                                    <td><?php echo esc_html(mysql2date(get_option('date_format') . ' ' . get_option('time_format'), $order->created_at)); ?></td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Cliente:', 'hng-commerce'); ?></th>
                                    <td>
                                        <?php 
                                        $customer_name = trim(($order->billing_first_name ?? '') . ' ' . ($order->billing_last_name ?? ''));
                                        if (empty($customer_name)) {
                                            $customer_name = $order->customer_name ?? __('N/A', 'hng-commerce');
                                        }
                                        $customer_email = $order->billing_email ?? ($order->customer_email ?? '');
                                        echo esc_html($customer_name); 
                                        ?><br>
                                        <small><?php echo esc_html($customer_email); ?></small>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Total:', 'hng-commerce'); ?></th>
                                    <td><strong><?php echo esc_html('R$ ' . number_format($order->total, 2, ',', '.')); ?></strong></td>
                                </tr>
                                <?php if (!empty($order->payment_method)): ?>
                                <tr>
                                    <th><?php esc_html_e('Método de Pagamento:', 'hng-commerce'); ?></th>
                                    <td><?php echo esc_html(ucfirst(str_replace('_', ' ', $order->payment_method))); ?></td>
                                </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <!-- Customer Address -->
                <div class="hng-card">
                    <div class="hng-card-header">
                        <h2 class="hng-card-title"><?php esc_html_e('Endereço de Entrega', 'hng-commerce'); ?></h2>
                    </div>
                    <div class="hng-card-content">
                        <?php
                        // Primeiro tenta os novos campos de billing
                        $has_new_billing = !empty($order->billing_address_1);
                        
                        if ($has_new_billing):
                        ?>
                            <address>
                                <?php
                                echo esc_html($order->billing_address_1);
                                if (!empty($order->billing_number)) {
                                    echo ', ' . esc_html($order->billing_number);
                                }
                                echo '<br>';
                                if (!empty($order->billing_address_2)) {
                                    echo esc_html($order->billing_address_2) . '<br>';
                                }
                                if (!empty($order->billing_neighborhood)) {
                                    echo esc_html($order->billing_neighborhood) . '<br>';
                                }
                                if (!empty($order->billing_city) && !empty($order->billing_state)) {
                                    echo esc_html($order->billing_city) . ' - ' . esc_html($order->billing_state) . '<br>';
                                }
                                if (!empty($order->billing_postcode)) {
                                    echo esc_html__('CEP:', 'hng-commerce') . ' ' . esc_html($order->billing_postcode) . '<br>';
                                }
                                if (!empty($order->billing_phone)) {
                                    echo esc_html__('Tel:', 'hng-commerce') . ' ' . esc_html($order->billing_phone) . '<br>';
                                }
                                if (!empty($order->billing_cpf)) {
                                    echo esc_html__('CPF:', 'hng-commerce') . ' ' . esc_html($order->billing_cpf);
                                }
                                ?>
                            </address>
                        <?php 
                        else:
                            // Fallback para endereço serializado antigo
                            $billing_address = maybe_unserialize($order->billing_address ?? '');
                            if (!empty($billing_address) && is_array($billing_address)):
                        ?>
                            <address>
                                <?php
                                if (!empty($billing_address['street'])) {
                                    echo esc_html($billing_address['street']);
                                    if (!empty($billing_address['number'])) {
                                        echo ', ' . esc_html($billing_address['number']);
                                    }
                                    echo '<br>';
                                }
                                if (!empty($billing_address['complement'])) {
                                    echo esc_html($billing_address['complement']) . '<br>';
                                }
                                if (!empty($billing_address['district'])) {
                                    echo esc_html($billing_address['district']) . '<br>';
                                }
                                if (!empty($billing_address['city']) && !empty($billing_address['state'])) {
                                    echo esc_html($billing_address['city']) . ' - ' . esc_html($billing_address['state']) . '<br>';
                                }
                                if (!empty($billing_address['postcode'])) {
                                    echo esc_html__('CEP:', 'hng-commerce') . ' ' . esc_html($billing_address['postcode']);
                                }
                                ?>
                            </address>
                        <?php else: ?>
                            <p><?php esc_html_e('Endereço não disponível.', 'hng-commerce'); ?></p>
                        <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <!-- Shipping & Label Section -->
            <?php
            $shipping_method = get_post_meta($order_id, '_shipping_method', true);
            $shipping_data = get_post_meta($order_id, '_shipping_data', true);
            $tracking_code = get_post_meta($order_id, '_hng_tracking_code', true);
            $label_data = get_post_meta($order_id, '_hng_shipping_label', true);
            $needs_shipping = false;
            
            // Verificar se algum item precisa de envio
            foreach ($items as $item) {
                $product_type = get_post_meta($item->product_id, '_hng_product_type', true);
                if (empty($product_type) || $product_type === 'physical') {
                    $needs_shipping = true;
                    break;
                }
            }
            
            if ($needs_shipping):
            ?>
            <div class="hng-card" style="margin-top: 20px;">
                <div class="hng-card-header" style="display: flex; justify-content: space-between; align-items: center;">
                    <h2 class="hng-card-title">
                        <span class="dashicons dashicons-car" style="vertical-align: middle;"></span>
                        <?php esc_html_e('Envio e Etiqueta', 'hng-commerce'); ?>
                    </h2>
                </div>
                <div class="hng-card-content">
                    <table class="widefat striped">
                        <tbody>
                            <?php if (!empty($shipping_method)): ?>
                            <tr>
                                <th style="width: 200px;"><?php esc_html_e('Método de Envio:', 'hng-commerce'); ?></th>
                                <td><?php echo esc_html(ucfirst(str_replace(['_', ':'], [' ', ' - '], $shipping_method))); ?></td>
                            </tr>
                            <?php endif; ?>
                            
                            <?php 
                            $shipping_cost = $order->shipping_total ?? ($order->shipping_cost ?? ($shipping_data['cost'] ?? 0));
                            if (!empty($shipping_cost)): 
                            ?>
                            <tr>
                                <th><?php esc_html_e('Custo do Frete:', 'hng-commerce'); ?></th>
                                <td><?php echo esc_html('R$ ' . number_format(floatval($shipping_cost), 2, ',', '.')); ?></td>
                            </tr>
                            <?php endif; ?>
                            
                            <tr>
                                <th><?php esc_html_e('Código de Rastreamento:', 'hng-commerce'); ?></th>
                                <td>
                                    <?php if (!empty($tracking_code)): ?>
                                        <code style="font-size: 14px; padding: 5px 10px; background: #f0f0f0; border-radius: 4px;"><?php echo esc_html($tracking_code); ?></code>
                                        <a href="https://www.linkcorreios.com.br/?id=<?php echo esc_attr($tracking_code); ?>" target="_blank" class="button button-small" style="margin-left: 10px;">
                                            <?php esc_html_e('Rastrear', 'hng-commerce'); ?>
                                        </a>
                                    <?php else: ?>
                                        <em><?php esc_html_e('Não disponível', 'hng-commerce'); ?></em>
                                        <input type="text" id="hng_tracking_code" placeholder="<?php esc_attr_e('Inserir código', 'hng-commerce'); ?>" style="width: 200px; margin-left: 10px;">
                                        <button type="button" class="button button-small" id="hng_save_tracking">
                                            <?php esc_html_e('Salvar', 'hng-commerce'); ?>
                                        </button>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            
                            <tr>
                                <th><?php esc_html_e('Etiqueta de Envio:', 'hng-commerce'); ?></th>
                                <td>
                                    <?php if (!empty($label_data)): ?>
                                        <?php if ($label_data['status'] === 'generated' && !empty($label_data['label_url'])): ?>
                                            <span class="dashicons dashicons-yes-alt" style="color: #46b450; vertical-align: middle;"></span>
                                            <strong style="color: #46b450;"><?php esc_html_e('Etiqueta Gerada', 'hng-commerce'); ?></strong>
                                            <a href="<?php echo esc_url($label_data['label_url']); ?>" target="_blank" class="button button-primary" style="margin-left: 15px;">
                                                <span class="dashicons dashicons-download" style="vertical-align: middle;"></span>
                                                <?php esc_html_e('Baixar Etiqueta', 'hng-commerce'); ?>
                                            </a>
                                        <?php elseif ($label_data['status'] === 'manual'): ?>
                                            <span class="dashicons dashicons-warning" style="color: #ffb900; vertical-align: middle;"></span>
                                            <?php echo esc_html($label_data['message'] ?? __('Geração manual necessária.', 'hng-commerce')); ?>
                                            <?php if (!empty($label_data['manual_url'])): ?>
                                                <a href="<?php echo esc_url($label_data['manual_url']); ?>" target="_blank" class="button" style="margin-left: 10px;">
                                                    <?php esc_html_e('Acessar Portal', 'hng-commerce'); ?>
                                                </a>
                                            <?php endif; ?>
                                        <?php elseif (!empty($label_data['portal_url'])): ?>
                                            <span class="dashicons dashicons-external" style="color: #0073aa; vertical-align: middle;"></span>
                                            <?php echo esc_html($label_data['message'] ?? ''); ?>
                                            <a href="<?php echo esc_url($label_data['portal_url']); ?>" target="_blank" class="button" style="margin-left: 10px;">
                                                <?php esc_html_e('Acessar Portal', 'hng-commerce'); ?>
                                            </a>
                                        <?php else: ?>
                                            <span class="dashicons dashicons-clock" style="color: #ffb900; vertical-align: middle;"></span>
                                            <?php echo esc_html($label_data['message'] ?? __('Processando...', 'hng-commerce')); ?>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <em><?php esc_html_e('Nenhuma etiqueta gerada', 'hng-commerce'); ?></em>
                                        <button type="button" class="button button-primary" id="hng_generate_label" style="margin-left: 15px;">
                                            <span class="dashicons dashicons-printer" style="vertical-align: middle;"></span>
                                            <?php esc_html_e('Gerar Etiqueta', 'hng-commerce'); ?>
                                        </button>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
            
            <script>
            jQuery(document).ready(function($) {
                // Gerar etiqueta
                $('#hng_generate_label').on('click', function() {
                    var $btn = $(this);
                    $btn.prop('disabled', true).text('<?php echo esc_js(__('Gerando...', 'hng-commerce')); ?>');
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'hng_generate_shipping_label',
                            order_id: <?php echo (int) $order_id; ?>,
                            nonce: '<?php echo esc_js(wp_create_nonce('hng_shipping_label')); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                location.reload();
                            } else {
                                alert(response.data.message || '<?php echo esc_js(__('Erro ao gerar etiqueta.', 'hng-commerce')); ?>');
                                $btn.prop('disabled', false).html('<span class="dashicons dashicons-printer" style="vertical-align: middle;"></span> <?php echo esc_js(__('Gerar Etiqueta', 'hng-commerce')); ?>');
                            }
                        },
                        error: function() {
                            alert('<?php echo esc_js(__('Erro de conexão.', 'hng-commerce')); ?>');
                            $btn.prop('disabled', false).html('<span class="dashicons dashicons-printer" style="vertical-align: middle;"></span> <?php echo esc_js(__('Gerar Etiqueta', 'hng-commerce')); ?>');
                        }
                    });
                });
                
                // Salvar código de rastreamento
                $('#hng_save_tracking').on('click', function() {
                    var code = $('#hng_tracking_code').val().trim();
                    if (!code) {
                        alert('<?php echo esc_js(__('Digite um código de rastreamento.', 'hng-commerce')); ?>');
                        return;
                    }
                    
                    var $btn = $(this);
                    $btn.prop('disabled', true).text('<?php echo esc_js(__('Salvando...', 'hng-commerce')); ?>');
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'hng_save_tracking_code',
                            order_id: <?php echo (int) $order_id; ?>,
                            tracking_code: code,
                            nonce: '<?php echo esc_js(wp_create_nonce('hng_shipping_label')); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                location.reload();
                            } else {
                                alert(response.data.message || '<?php echo esc_js(__('Erro ao salvar.', 'hng-commerce')); ?>');
                                $btn.prop('disabled', false).text('<?php echo esc_js(__('Salvar', 'hng-commerce')); ?>');
                            }
                        },
                        error: function() {
                            alert('<?php echo esc_js(__('Erro de conexão.', 'hng-commerce')); ?>');
                            $btn.prop('disabled', false).text('<?php echo esc_js(__('Salvar', 'hng-commerce')); ?>');
                        }
                    });
                });
            });
            </script>
            <?php endif; ?>
            
            <!-- Order Items -->
            <div class="hng-card" style="margin-top: 20px;">
                <div class="hng-card-header">
                    <h2 class="hng-card-title"><?php esc_html_e('Itens do Pedido', 'hng-commerce'); ?></h2>
                </div>
                <div class="hng-card-content">
                    <?php if (!empty($items)): ?>
                        <table class="widefat striped">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e('Produto', 'hng-commerce'); ?></th>
                                    <th><?php esc_html_e('Quantidade', 'hng-commerce'); ?></th>
                                    <th><?php esc_html_e('Preço Unit.', 'hng-commerce'); ?></th>
                                    <th><?php esc_html_e('Subtotal', 'hng-commerce'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($items as $item): ?>
                                    <tr>
                                        <td><?php echo esc_html($item->product_name); ?></td>
                                        <td><?php echo esc_html($item->quantity); ?></td>
                                        <td><?php echo esc_html('R$ ' . number_format($item->price, 2, ',', '.')); ?></td>
                                        <td><?php echo esc_html('R$ ' . number_format($item->price * $item->quantity, 2, ',', '.')); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot>
                                <tr>
                                    <th colspan="3" style="text-align: right;"><?php esc_html_e('Total:', 'hng-commerce'); ?></th>
                                    <th><?php echo esc_html('R$ ' . number_format($order->total, 2, ',', '.')); ?></th>
                                </tr>
                            </tfoot>
                        </table>
                    <?php else: ?>
                        <p><?php esc_html_e('Nenhum item encontrado neste pedido.', 'hng-commerce'); ?></p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Handle bulk actions
     */
    private static function handle_bulk_actions() {
        // Verify nonce
        if (!isset($_POST['hng_orders_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['hng_orders_nonce'])), 'hng_bulk_orders')) {
            wp_die(esc_html__('Erro de segurança. Por favor, tente novamente.', 'hng-commerce'));
        }
        
        // Get bulk action
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
        $bulk_action = isset($_POST['action']) && $_POST['action'] !== '-1' ? sanitize_text_field(wp_unslash($_POST['action'])) : (isset($_POST['action2']) ? sanitize_text_field(wp_unslash($_POST['action2'])) : '');
        
        if (empty($bulk_action)) {
            return;
        }
        
        // Get selected orders
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
        $order_ids = isset($_POST['order']) ? array_map('absint', wp_unslash($_POST['order'])) : array();
        
        if (empty($order_ids)) {
            return;
        }
        
        global $wpdb;
        $orders_table = hng_db_full_table_name('hng_orders');
        $orders_table_sql = '`' . str_replace('`', '', $orders_table) . '`';
        
        // Map bulk actions to statuses
        $status_map = array(
            'mark_processing' => 'processing',
            'mark_completed' => 'completed',
            'mark_cancelled' => 'cancelled',
        );
        
        if (isset($status_map[$bulk_action])) {
            $new_status = $status_map[$bulk_action];
            
            foreach ($order_ids as $order_id) {
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Bulk update with sanitized table name
                // phpcs:ignore PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name sanitized via hng_db_full_table_name and backtick escaping
                $wpdb->update(
                    $orders_table,
                    array('status' => $new_status),
                    array('id' => $order_id),
                    array('%s'),
                    array('%d')
                );
            }
            
            // Show success message
            add_action('admin_notices', function() use ($order_ids, $new_status) {
                echo '<div class="notice notice-success is-dismissible"><p>';
                /* translators: 1: Number of orders updated, 2: New status */
                echo esc_html(sprintf(__('%1$d pedido(s) atualizado(s) para status: %2$s', 'hng-commerce'), count($order_ids), $new_status));
                echo '</p></div>';
            });
        }
    }
}
