<?php
/**
 * HNG Admin - Quotes Page
 *
 * Página de gerenciamento de orçamentos pendentes de aprovação.
 *
 * @package HNG_Commerce
 * @subpackage Admin/Pages
 * @since 1.1.1
 */

if (!defined('ABSPATH')) {
    exit;
}

class HNG_Admin_Quotes_Page {
    
    /**
     * Renderizar página
     */
    public static function render() {
        global $wpdb;
        
        $breadcrumbs = [
            ['text' => 'HNG Commerce', 'url' => admin_url('admin.php?page=hng-commerce')],
            ['text' => __('Orçamentos', 'hng-commerce')]
        ];
        
        // Buscar orçamentos pendentes
        $orders_table_sql = hng_db_backtick_table('hng_orders');
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $orders_table_sql sanitized via hng_db_backtick_table()
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query for quotes management, pending approval orders
        // phpcs:ignore PluginCheck.Security.DirectDB.UnescapedDBParameter -- $orders_table_sql sanitized via hng_db_backtick_table()
        $quotes = $wpdb->get_results(
            "SELECT * FROM {$orders_table_sql} 
            WHERE status IN ('hng-pending-approval', 'hng-awaiting-payment')
            ORDER BY created_at DESC"
        );
        
        self::render_header($breadcrumbs);
        
        if (empty($quotes)) {
            self::render_empty_state();
        } else {
            self::render_quotes_table($quotes);
            self::render_tip();
        }
        
        self::render_styles();
    }
    
    /**
     * Renderizar cabeçalho
     */
    private static function render_header($breadcrumbs) {
        echo '<div class="hng-wrap">';
        self::render_breadcrumbs($breadcrumbs);
        
        echo '<div class="hng-header">';
        echo '<h1>';
        echo '<span class="dashicons dashicons-clipboard"></span>';
        echo esc_html__('Gerenciamento de Orçamentos', 'hng-commerce');
        echo '</h1>';
        echo '</div>';
        
        echo '<div class="hng-content">';
    }
    
    /**
     * Renderizar breadcrumbs
     */
    private static function render_breadcrumbs($items) {
        if (empty($items)) {
            return;
        }
        
        echo '<nav class="hng-breadcrumbs">';
        foreach ($items as $i => $item) {
            if ($i > 0) {
                echo ' <span class="separator">/</span> ';
            }
            
            if (isset($item['url'])) {
                echo '<a href="' . esc_url($item['url']) . '">' . esc_html($item['text']) . '</a>';
            } else {
                echo '<span>' . esc_html($item['text']) . '</span>';
            }
        }
        echo '</nav>';
    }
    
    /**
     * Renderizar estado vazio
     */
    private static function render_empty_state() {
        echo '<div class="hng-empty-state">';
        echo '<span class="dashicons dashicons-clipboard" style="font-size: 80px; opacity: 0.2;"></span>';
        echo '<h2>' . esc_html__('Nenhum orçamento pendente', 'hng-commerce') . '</h2>';
        echo '<p>' . esc_html__('Quando clientes solicitarem orçamentos, eles aparecerão aqui.', 'hng-commerce') . '</p>';
        echo '</div>';
    }
    
    /**
     * Renderizar tabela de orçamentos
     */
    private static function render_quotes_table($quotes) {
        echo '<table class="wp-list-table widefat fixed">';
        echo '<thead>';
        echo '<tr>';
        echo '<th>' . esc_html__('Pedido', 'hng-commerce') . '</th>';
        echo '<th>' . esc_html__('Cliente', 'hng-commerce') . '</th>';
        echo '<th>' . esc_html__('Status', 'hng-commerce') . '</th>';
        echo '<th>' . esc_html__('Valor Solicitado', 'hng-commerce') . '</th>';
        echo '<th>' . esc_html__('Data', 'hng-commerce') . '</th>';
        echo '<th>' . esc_html__('Ações', 'hng-commerce') . '</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody>';
        
        foreach ($quotes as $quote) {
            self::render_quote_row($quote);
        }
        
        echo '</tbody>';
        echo '</table>';
    }
    
    /**
     * Renderizar linha de orçamento
     */
    private static function render_quote_row($quote) {
        $status_labels = [
            'hng-pending-approval' => __('Aguardando Aprovação', 'hng-commerce'),
            'hng-awaiting-payment' => __('Aguardando Pagamento', 'hng-commerce'),
        ];
        
        $status_colors = [
            'hng-pending-approval' => '#f0ad4e',
            'hng-awaiting-payment' => '#5bc0de',
        ];
        
        $status_label = $status_labels[$quote->status] ?? $quote->status;
        $status_color = $status_colors[$quote->status] ?? '#666';
        
        echo '<tr>';
        
        // Pedido
        echo '<td>';
        echo '<strong>#' . esc_html($quote->order_number) . '</strong>';
        echo '</td>';
        
        // Cliente
        echo '<td>';
        echo esc_html($quote->customer_name) . '<br>';
        echo '<small>' . esc_html($quote->customer_email) . '</small>';
        echo '</td>';
        
        // Status
        echo '<td>';
        echo '<span style="display: inline-block; padding: 4px 10px; border-radius: 3px; background: ' . esc_attr($status_color) . '; color: white; font-size: 11px; font-weight: bold;">';
        echo esc_html($status_label);
        echo '</span>';
        echo '</td>';
        
        // Valor
        echo '<td>';
        echo '<strong>' . esc_html(hng_price($quote->total)) . '</strong>';
        echo '</td>';
        
        // Data
        echo '<td>';
        echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($quote->created_at)));
        echo '</td>';
        
        // Ações
        echo '<td>';
        $post_id = $quote->post_id ?? 0;
        if ($post_id) {
            echo '<a href="' . esc_url(admin_url('post.php?post=' . $post_id . '&action=edit')) . '" class="button button-primary button-small">';
            if ($quote->status === 'hng-pending-approval') {
                echo '<span class="dashicons dashicons-yes-alt" style="margin-top: 3px;"></span>';
                echo esc_html__('Aprovar', 'hng-commerce');
            } else {
                echo '<span class="dashicons dashicons-visibility" style="margin-top: 3px;"></span>';
                echo esc_html__('Ver Detalhes', 'hng-commerce');
            }
            echo '</a>';
        }
        echo '</td>';
        
        echo '</tr>';
    }
    
    /**
     * Renderizar dica
     */
    private static function render_tip() {
        echo '<div style="margin-top: 20px; padding: 15px; background: #f0f6fc; border-left: 4px solid #2271b1; border-radius: 4px;">';
        echo '<p style="margin: 0;">';
        echo '<strong>' . esc_html__('💡 Dica:', 'hng-commerce') . '</strong> ';
        echo esc_html__('Clique em "Aprovar" para definir o preço final e gerar o link de pagamento para o cliente.', 'hng-commerce');
        echo '</p>';
        echo '</div>';
        
        echo '</div>'; // .hng-content
        echo '</div>'; // .hng-wrap
    }
    
    /**
     * Renderizar estilos
     */
    private static function render_styles() {
        ?>
        <style>
        .hng-empty-state {
            text-align: center;
            padding: 80px 20px;
            background: #fff;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .hng-empty-state h2 {
            margin: 20px 0 10px 0;
            color: #666;
        }
        .hng-empty-state p {
            color: #999;
            font-size: 14px;
        }
        .hng-breadcrumbs {
            margin-bottom: 20px;
            font-size: 13px;
            color: #666;
        }
        .hng-breadcrumbs a {
            color: #2271b1;
            text-decoration: none;
        }
        .hng-breadcrumbs a:hover {
            text-decoration: underline;
        }
        .hng-breadcrumbs .separator {
            margin: 0 5px;
            color: #ccc;
        }
        .hng-header h1 {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        </style>
        <?php
    }
}
