<?php
/**
 * AJAX Handlers
 * 
 * @package HNG_Commerce
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class HNG_Ajax {
    
    /**
     * Instância única
     */
    private static $instance = null;
    
    /**
     * Obter instância
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Construtor
     */
    private function __construct() {
        // Adicionar ao carrinho
        add_action('wp_ajax_hng_add_to_cart', [$this, 'add_to_cart']);
        add_action('wp_ajax_nopriv_hng_add_to_cart', [$this, 'add_to_cart']);

        // REST fallback para ambientes com bloqueio no admin-ajax.php (ex: WAF)
        add_action('rest_api_init', [$this, 'register_rest_routes']);
        
        // Remover do carrinho
        add_action('wp_ajax_hng_remove_from_cart', [$this, 'remove_from_cart']);
        add_action('wp_ajax_nopriv_hng_remove_from_cart', [$this, 'remove_from_cart']);
        
        // Atualizar quantidade
        add_action('wp_ajax_hng_update_cart_quantity', [$this, 'update_cart_quantity']);
        add_action('wp_ajax_nopriv_hng_update_cart_quantity', [$this, 'update_cart_quantity']);
        
        // Aplicar cupom
        add_action('wp_ajax_hng_apply_coupon', [$this, 'apply_coupon']);
        add_action('wp_ajax_nopriv_hng_apply_coupon', [$this, 'apply_coupon']);
        
        // Remover cupom
        add_action('wp_ajax_hng_remove_coupon', [$this, 'remove_coupon']);
        add_action('wp_ajax_nopriv_hng_remove_coupon', [$this, 'remove_coupon']);
        
        // Calcular frete
        add_action('wp_ajax_hng_calculate_shipping', [$this, 'calculate_shipping']);
        add_action('wp_ajax_nopriv_hng_calculate_shipping', [$this, 'calculate_shipping']);
        
        // Atualizar frete do carrinho
        add_action('wp_ajax_hng_update_cart_shipping', [$this, 'update_cart_shipping']);
        
        // Rastreamento: Buscar atualizações
        add_action('wp_ajax_hng_check_tracking', [$this, 'check_tracking']);
        add_action('wp_ajax_nopriv_hng_check_tracking', [$this, 'check_tracking']);
        
        // Processar pagamento com cartão
        add_action('wp_ajax_hng_process_card_payment', [$this, 'process_card_payment']);
        add_action('wp_ajax_nopriv_hng_process_card_payment', [$this, 'process_card_payment']);
        // Toggle gateway enable/disable

        // Configurar gateway

        
        // Import / Export tools (admin only)
        add_action('wp_ajax_hng_upload_csv', [$this, 'upload_csv']);
        add_action('wp_ajax_hng_import_woocommerce', [$this, 'import_woocommerce']);
        add_action('wp_ajax_hng_export_products', [$this, 'export_products']);
        add_action('wp_ajax_hng_export_orders', [$this, 'export_orders']);
        
        // Run SQL Import (System Tool)
        add_action('wp_ajax_hng_run_sql_import', [$this, 'run_sql_import']);
        
        // WP-Cron batch processor for imports
        add_action('hng_process_import_batch', [$this, 'process_import_batch'], 10, 1);
        
    }
    
    /**
     * Executes an uploaded SQL file
     */
    public function run_sql_import() {
        if (!check_ajax_referer('hng-commerce-admin', 'nonce', false)) {
            wp_send_json_error(['message' => __('Segurança inválida.', 'hng-commerce')]);
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Sem permissão.', 'hng-commerce')]);
        }

        if (empty($_FILES['file'])) {
            wp_send_json_error(['message' => __('Arquivo não enviado.', 'hng-commerce')]);
        }

        $file = $_FILES['file'];
        $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        if (strtolower($ext) !== 'sql') {
            wp_send_json_error(['message' => __('Apenas arquivos .sql são permitidos.', 'hng-commerce')]);
        }

        // Use WP_Filesystem for reading uploaded file
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }
        $content = $wp_filesystem->get_contents($file['tmp_name']);
        if (!$content) {
            wp_send_json_error(['message' => __('Erro ao ler arquivo SQL.', 'hng-commerce')]);
        }

        global $wpdb;
        
        // Split SQL by semicolon, but handle cases where semicolons are inside strings if possible.
        // For simplicity, we assume standard dump format where statements end with ;
        // We can use WPDB's query method for each valid statement.
        
        // Remove comments
        $content = preg_replace('/^--.*$/m', '', $content);
        $content = preg_replace('/^\/\*.*\*\/$/m', '', $content);
        
        $queries = explode(';', $content);
        $executed = 0;
        $errors = 0;

        foreach ($queries as $query) {
            $query = trim($query);
            if (empty($query)) continue;
            
            // Replace prefix placeholder if exists (e.g. wp_) with current db prefix
            // This is optional but good for portability. 
            // For now, assume SQL has correct headers or generic creates.
            
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Dynamic SQL from uploaded file, cannot use prepare() for arbitrary SQL statements
            $result = $wpdb->query($query);
            if ($result === false) {
                // Log error but continue
                error_log('HNG SQL Import Error: ' . $wpdb->last_error . ' | Query: ' . substr($query, 0, 100));
                $errors++;
            } else {
                $executed++;
            }
        }

        wp_send_json_success([
            /* translators: 1: number of successful commands, 2: number of errors */
            'message' => sprintf(__('SQL Executado! Comandos bem-sucedidos: %1$d. Erros: %2$d.', 'hng-commerce'), $executed, $errors),
            'executed' => $executed,
            'errors' => $errors
        ]);
    }

    /**
     * Adicionar ao carrinho
     */
    public function add_to_cart() {
        // Verify nonce from frontend
        if (!check_ajax_referer('hng_add_to_cart', 'nonce', false) && !check_ajax_referer('hng_cart_actions', 'nonce', false)) {
             // DEBUG: Extended info with forced JSON header if needed, but for now just return clean error
             wp_send_json_error(['message' => __('Sessão expirada. Recarregue a página.', 'hng-commerce')], 403);
        }

        $result = $this->process_add_to_cart_request($_POST);
        if ($result['success']) {
            wp_send_json_success($result['data']);
        }
        wp_send_json_error($result['data']);
    }



    /**
     * REST fallback para add_to_cart
     */
    public function rest_add_to_cart($request) {
        $params = $request->get_json_params();
        if (empty($params)) {
            $params = $request->get_params();
        }
        $result = $this->process_add_to_cart_request($params);
        if ($result['success']) {
            return new WP_REST_Response($result['data'], 200);
        }
        return new WP_REST_Response($result['data'], 400);
    }

    public function register_rest_routes() {
        register_rest_route(
            'hng/v1',
            '/add-to-cart',
            [
                'methods'  => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'rest_add_to_cart'],
                'permission_callback' => '__return_true',
            ]
        );

        register_rest_route(
            'hng/v1',
            '/calculate-shipping',
            [
                'methods'  => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'rest_calculate_shipping'],
                'permission_callback' => '__return_true',
            ]
        );
    }

    /**
     * REST API callback para cálculo de frete (evita bloqueios WAF no admin-ajax)
     */
    public function rest_calculate_shipping($request) {
        $params = $request->get_json_params();
        if (empty($params)) {
            $params = $request->get_params();
        }

        $postcode = preg_replace('/\D/', '', (string) ($params['postcode'] ?? ''));
        if (strlen($postcode) !== 8) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('CEP inválido. Informe 8 dígitos.', 'hng-commerce'),
            ], 400);
        }

        $manager = HNG_Shipping_Manager::instance();
        $product_id = absint($params['product_id'] ?? 0);
        $quantity = max(1, absint($params['quantity'] ?? 1));

        // Se product_id foi informado, calcular para produto específico
        if ($product_id > 0) {
            $package = $manager->build_package_from_product($product_id, $quantity, $postcode);
        } else {
            // Senão, calcular para o carrinho atual
            $cart = hng_cart();
            if ($cart->is_empty()) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => __('Carrinho vazio. Adicione produtos para calcular o frete.', 'hng-commerce'),
                ], 400);
            }
            $package = $manager->build_package_from_cart($postcode);
        }

        $rates = $manager->calculate_shipping($package);

        if (is_wp_error($rates)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $rates->get_error_message(),
            ], 400);
        }

        if (empty($rates)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Nenhuma opção de frete disponível para este CEP.', 'hng-commerce'),
            ], 200);
        }

        $methods = array_map(static function ($rate) {
            $delivery_label = $rate['delivery_time_label'] ?? '';
            return [
                'id' => $rate['id'] ?? '',
                'method_id' => $rate['method_id'] ?? '',
                'service' => $rate['service_name'] ?? $rate['method_title'] ?? '',
                'name' => $rate['service_name'] ?? $rate['method_title'] ?? '',
                'label' => $rate['label'] ?? '',
                'cost' => floatval($rate['cost'] ?? 0),
                'delivery_time' => $delivery_label,
                'delivery_time_text' => $delivery_label,
                'delivery_time_label' => $delivery_label,
                'raw_delivery_days' => $rate['delivery_time'] ?? 0,
            ];
        }, $rates);

        return new WP_REST_Response([
            'success' => true,
            'methods' => $methods,
        ], 200);
    }

    private function process_add_to_cart_request($raw) {
        // Debug apenas em desenvolvimento
        if (defined('HNG_DEBUG') && HNG_DEBUG === true) {
            error_log('HNG Debug: process_add_to_cart_request - ' . wp_json_encode($raw));
        }

        $post = wp_unslash($raw);
        
        // Validação de entrada rigorosa - SECURITY FIX
        $product_id = absint($post['product_id'] ?? 0);
        $quantity = absint($post['quantity'] ?? 1);
        
        // Validar range de quantidade
        if ($quantity < 1) {
            return [
                'success' => false,
                'data' => ['message' => __('Quantidade deve ser maior que zero.', 'hng-commerce')]
            ];
        }
        
        if ($quantity > 1000) {
            return [
                'success' => false,
                'data' => ['message' => __('Quantidade máxima por item é 1000.', 'hng-commerce')]
            ];
        }
        
        if ($product_id < 1) {
            return [
                'success' => false,
                'data' => ['message' => __('ID de produto inválido.', 'hng-commerce')]
            ];
        }
        $variation_id = absint($post['variation_id'] ?? 0);
        $variation = isset($post['variation']) ? (array) $post['variation'] : [];

        if (empty($product_id)) {
            return [
                'success' => false,
                'data' => ['message' => __('Produto inválido.', 'hng-commerce')]
            ];
        }

        $product = hng_get_product($product_id);

        if (!$product->get_id()) {
            return [
                'success' => false,
                'data' => ['message' => __('Produto não encontrado.', 'hng-commerce')]
            ];
        }

        if (!$product->is_purchasable()) {
            return [
                'success' => false,
                'data' => ['message' => __('Este produto não pode ser comprado.', 'hng-commerce')]
            ];
        }

        if (!$product->is_in_stock()) {
            return [
                'success' => false,
                'data' => ['message' => __('Produto fora de estoque.', 'hng-commerce')]
            ];
        }

        if ($product->manages_stock() && $quantity > $product->get_stock_quantity()) {
            return [
                'success' => false,
                /* translators: %d: available stock quantity */
                'data' => ['message' => sprintf(esc_html__('Quantidade indisponível. Estoque: %d unidades.', 'hng-commerce'), $product->get_stock_quantity())]
            ];
        }

        $custom_fields = [];
        if (!empty($post['hng_cf']) && is_array($post['hng_cf'])) {
            foreach ($post['hng_cf'] as $slug => $value) {
                $custom_fields[$slug] = sanitize_text_field($value);
            }
        }

        $cart = hng_cart();
        $variation['custom_fields'] = $custom_fields;
        $cart_id = $cart->add_to_cart($product_id, $quantity, $variation_id, $variation);

        if ($cart_id) {
            if (!isset($_SESSION)) {
                session_start();
            }
            if (!isset($_SESSION['hng_cf_checkout'])) {
                $_SESSION['hng_cf_checkout'] = [];
            }
            $_SESSION['hng_cf_checkout'][$cart_id] = $custom_fields;

            return [
                'success' => true,
                'data' => [
                    'message' => __('Produto adicionado ao carrinho!', 'hng-commerce'),
                    'cart_count' => $cart->get_cart_contents_count(),
                    'cart_subtotal' => hng_price($cart->get_subtotal()),
                    'cart_total' => hng_price($cart->get_total()),
                    'cart_hash' => md5(wp_json_encode($cart->get_cart())),
                ],
            ];
        }

        return [
            'success' => false,
            'data' => ['message' => __('Erro ao adicionar produto ao carrinho.', 'hng-commerce')]
        ];
    }


    /**
     * Calcular frete usando métodos HTTPS (Correios, Melhor Envio, Jadlog)
     */
    public function calculate_shipping() {
        $nonce = wp_unslash($_POST['nonce'] ?? '');
        $nonce_ok = check_ajax_referer('hng_calculate_shipping', 'nonce', false) || check_ajax_referer('hng_cart_actions', 'nonce', false);

        if (!$nonce_ok) {
            wp_send_json_error([
                'message' => __('Erro de segurança. Recarregue a página e tente novamente.', 'hng-commerce'),
            ], 403);
        }

        $postcode = preg_replace('/\D/', '', (string) ($_POST['postcode'] ?? ''));
        if (strlen($postcode) !== 8) {
            wp_send_json_error([
                'message' => __('CEP inválido. Informe 8 dígitos.', 'hng-commerce'),
            ]);
        }

        $cart = hng_cart();
        if ($cart->is_empty()) {
            wp_send_json_error([
                'message' => __('Carrinho vazio. Adicione produtos para calcular o frete.', 'hng-commerce'),
            ]);
        }

        $manager = HNG_Shipping_Manager::instance();
        $package = $manager->build_package_from_cart($postcode);
        $rates = $manager->calculate_shipping($package);

        if (is_wp_error($rates)) {
            wp_send_json_error([
                'message' => $rates->get_error_message(),
            ]);
        }

        if (empty($rates)) {
            wp_send_json_error([
                'message' => __('Nenhuma opção de frete disponível para este CEP.', 'hng-commerce'),
            ]);
        }

        // Persistir cotações para seleção posterior
        $cart->set_available_shipping_rates($postcode, $rates);

        $methods = array_map(static function ($rate) {
            $delivery_label = $rate['delivery_time_label'] ?? '';
            return [
                'id' => $rate['id'] ?? '',
                'method_id' => $rate['method_id'] ?? '',
                'service' => $rate['service_name'] ?? $rate['method_title'] ?? '',
                'name' => $rate['service_name'] ?? $rate['method_title'] ?? '',
                'label' => $rate['label'] ?? '',
                'cost' => floatval($rate['cost'] ?? 0),
                'delivery_time' => $delivery_label,
                'delivery_time_text' => $delivery_label,
                'delivery_time_label' => $delivery_label,
                'raw_delivery_days' => $rate['delivery_time'] ?? 0,
            ];
        }, $rates);

        wp_send_json_success([
            'methods' => $methods,
        ]);
    }

    /**
     * Salvar método de frete escolhido no carrinho
     */
    public function update_cart_shipping() {
        $nonce = wp_unslash($_POST['nonce'] ?? '');
        if (!check_ajax_referer('hng_update_cart_shipping', 'nonce', false)) {
            wp_send_json_error([
                'message' => __('Erro de segurança. Recarregue a página e tente novamente.', 'hng-commerce'),
            ], 403);
        }

        $selected_id = sanitize_text_field($_POST['method_id'] ?? '');
        if (empty($selected_id)) {
            wp_send_json_error([
                'message' => __('Método de frete inválido.', 'hng-commerce'),
            ]);
        }

        $cart = hng_cart();
        $available = $cart->get_available_shipping_rates();
        $has_quotes = !empty($available['rates']);

        if (!$has_quotes) {
            wp_send_json_error([
                'message' => __('Recalcule o frete antes de selecionar um método.', 'hng-commerce'),
            ]);
        }

        if (!$cart->select_shipping_rate($selected_id)) {
            wp_send_json_error([
                'message' => __('Método de frete não encontrado. Recalcule o frete.', 'hng-commerce'),
            ]);
        }

        // Persist immediately to session
        $cart->save_cart();

        wp_send_json_success([
            'shipping_total' => hng_price($cart->get_shipping_total()),
            'cart_total' => hng_price($cart->get_total()),
            'selected' => $cart->get_selected_shipping(),
        ]);
    }

    /**
     * Remover item do carrinho
     */
    public function remove_from_cart() {
        if (!check_ajax_referer('hng_cart_actions', 'nonce', false)) {
            wp_send_json_error([
                'message' => __('Sessão expirada. Recarregue a página.', 'hng-commerce'),
            ], 403);
        }

        $cart_id = sanitize_text_field($_POST['cart_id'] ?? '');
        
        if (empty($cart_id)) {
            wp_send_json_error([
                'message' => __('ID do item inválido.', 'hng-commerce'),
            ]);
        }

        $cart = hng_cart();
        $result = $cart->remove($cart_id);

        if ($result) {
            wp_send_json_success([
                'message' => __('Produto removido do carrinho.', 'hng-commerce'),
                'cart_count' => $cart->get_cart_contents_count(),
                'cart_subtotal' => hng_price($cart->get_subtotal()),
                'cart_total' => hng_price($cart->get_total()),
            ]);
        }

        wp_send_json_error([
            'message' => __('Erro ao remover produto.', 'hng-commerce'),
        ]);
    }

    /**
     * Atualizar quantidade de item no carrinho
     */
    public function update_cart_quantity() {
        if (!check_ajax_referer('hng_cart_actions', 'nonce', false)) {
            wp_send_json_error([
                'message' => __('Sessão expirada. Recarregue a página.', 'hng-commerce'),
            ], 403);
        }

        $cart_id = sanitize_text_field($_POST['cart_id'] ?? '');
        $quantity = absint($_POST['quantity'] ?? 1);
        
        if (empty($cart_id)) {
            wp_send_json_error([
                'message' => __('ID do item inválido.', 'hng-commerce'),
            ]);
        }

        // Validação de range para prevenir ataques de quantidade
        if ($quantity < 1) {
            wp_send_json_error([
                'message' => __('Quantidade mínima é 1.', 'hng-commerce'),
            ]);
        }
        
        if ($quantity > 1000) {
            wp_send_json_error([
                'message' => __('Quantidade máxima é 1000 por item.', 'hng-commerce'),
            ]);
        }

        $cart = hng_cart();
        $result = $cart->set_quantity($cart_id, $quantity);

        if ($result) {
            $cart_contents = $cart->get_cart();
            $item_subtotal = 0;
            
            if (isset($cart_contents[$cart_id])) {
                $item = $cart_contents[$cart_id];
                $item_subtotal = $item['data']->get_price() * $item['quantity'];
            }

            wp_send_json_success([
                'message' => __('Quantidade atualizada.', 'hng-commerce'),
                'item_subtotal' => hng_price($item_subtotal),
                'cart_count' => $cart->get_cart_contents_count(),
                'cart_subtotal' => hng_price($cart->get_subtotal()),
                'cart_total' => hng_price($cart->get_total()),
            ]);
        }

        wp_send_json_error([
            'message' => __('Erro ao atualizar quantidade.', 'hng-commerce'),
        ]);
    }

    /**
     * Upload CSV (Import tools)
     */
    public function upload_csv() {
        if (!check_ajax_referer('hng-commerce-admin', 'nonce', false)) {
            wp_send_json_error(['message' => __('Erro de segurança. Recarregue a página e tente novamente.', 'hng-commerce')]);
        }

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- File upload validated by wp_handle_upload below
        if (empty($_FILES['file'])) {
            wp_send_json_error(['message' => __('Arquivo não enviado.', 'hng-commerce')]);
        }

        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';

        $overrides = ['test_form' => false, 'mimes' => ['csv' => 'text/csv', 'txt' => 'text/plain']];
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- File validated by wp_handle_upload
        $file = wp_handle_upload($_FILES['file'], $overrides);

        if (isset($file['error'])) {
            wp_send_json_error(['message' => $file['error']]);
        }

        // Inserir como attachment para facilitar uso posterior
        $file_path = $file['file'];
        $file_name = basename($file_path);
        $filetype = wp_check_filetype($file_name, null);
        $attachment = [
            'post_mime_type' => $filetype['type'] ?? 'text/csv',
            'post_title'     => sanitize_file_name(pathinfo($file_name, PATHINFO_FILENAME)),
            'post_content'   => '',
            'post_status'    => 'inherit'
        ];

        $attach_id = wp_insert_attachment($attachment, $file_path);
        if (!is_wp_error($attach_id) && $attach_id) {
            $attach_data = wp_generate_attachment_metadata($attach_id, $file_path);
            wp_update_attachment_metadata($attach_id, $attach_data);
            // salvar como último CSV carregado para importação
            update_option('hng_last_uploaded_csv', $attach_id);
            wp_send_json_success(['message' => __('Arquivo carregado com sucesso.', 'hng-commerce'), 'attachment_id' => $attach_id, 'url' => $file['url']]);
        }

        // fallback
        wp_send_json_success(['message' => __('Arquivo carregado com sucesso (não foi possível registrar attachment).', 'hng-commerce'), 'file' => $file]);
    }

    /**
     * Iniciar importação do WooCommerce (stub/processo em background)
     */
    public function import_woocommerce() {
        if (!check_ajax_referer('hng-commerce-admin', 'nonce', false)) {
            wp_send_json_error(['message' => __('Erro de segurança.', 'hng-commerce')]);
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Sem permissão.', 'hng-commerce')]);
        }

        $attachment_id = absint($_POST['attachment_id'] ?? 0);
        $import_type = sanitize_text_field($_POST['import_type'] ?? 'products');
        
        if (!$attachment_id) $attachment_id = get_option('hng_last_uploaded_csv');
        if (!$attachment_id) wp_send_json_error(['message' => __('Nenhum arquivo CSV encontrado.', 'hng-commerce')]);

        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            wp_send_json_error(['message' => __('Arquivo não encontrado no servidor.', 'hng-commerce')]);
        }

        // Parse CSV
        $file = new SplFileObject($file_path);
        $file->setFlags(SplFileObject::READ_CSV | SplFileObject::READ_AHEAD | SplFileObject::SKIP_EMPTY | SplFileObject::DROP_NEW_LINE);
        
        $headers = $file->fgetcsv();
        if (empty($headers)) {
            wp_send_json_error(['message' => __('Arquivo CSV vazio ou inválido.', 'hng-commerce')]);
        }

        // Map headers
        $map = [];
        foreach ($headers as $i => $h) {
            $k = strtolower(trim((string)$h));
            // Cleanup BOM if present
            if (strpos($k, "\xEF\xBB\xBF") === 0) $k = substr($k, 3);
            $k = preg_replace('/[^a-z0-9_]/', '_', $k);
            $map[$i] = $k;
        }

        $processed = 0;
        $created = 0;
        $updated = 0;
        $errors = 0;

        // Process Loop
        while (!$file->eof() && ($row = $file->fgetcsv()) !== null) {
            if (empty($row) || (count($row) === 1 && $row[0] === null)) continue;
            
            $data = [];
            foreach ($row as $i => $val) {
                if (isset($map[$i])) {
                    $data[$map[$i]] = trim($val);
                }
            }

            if ($import_type === 'orders') {
                $res = $this->_import_single_order($data);
            } else {
                $res = $this->_import_single_product($data);
            }
            
            if (is_wp_error($res)) {
                $errors++;
            } else {
                if ($res === 'updated') $updated++; else $created++;
            }
            $processed++;
        }

        wp_send_json_success([
            'message' => 'Importação finalizada',
            'imported' => array_fill(0, $created, 1), // Stub for frontend length check
            'updated' => array_fill(0, $updated, 1),
            'errors' => array_fill(0, $errors, 1)
        ]);
    }

    private function _import_single_product($data) {
        // Simplified Logic: Title, Price, Stock, SKU
        $sku = $data['sku'] ?? '';
        $name = $data['name'] ?? ($data['product_name'] ?? '');
        $price = $data['price'] ?? ($data['regular_price'] ?? '');
        
        if (empty($name) && empty($sku)) return new WP_Error('invalid', 'Sem nome ou sku');

        $post_id = 0;
        if ($sku) {
            $found = get_posts(['post_type' => 'hng_product', 'meta_key' => '_sku', 'meta_value' => $sku, 'fields' => 'ids']);
            if ($found) $post_id = $found[0];
        }

        $args = [
            'post_title' => $name,
            'post_type' => 'hng_product',
            'post_status' => 'publish'
        ];
        
        // Optional description
        if (!empty($data['description'])) $args['post_content'] = $data['description'];

        $is_update = false;
        if ($post_id) {
            $args['ID'] = $post_id;
            wp_update_post($args);
            $is_update = true;
        } else {
            $post_id = wp_insert_post($args);
            if (is_wp_error($post_id)) return $post_id;
        }

        // Meta
        if ($sku) update_post_meta($post_id, '_sku', $sku);
        if ($price) {
            $price = str_replace(',', '.', $price);
            update_post_meta($post_id, '_price', $price);
            update_post_meta($post_id, '_regular_price', $price);
        }
        if (isset($data['stock'])) {
            update_post_meta($post_id, '_stock_quantity', (int)$data['stock']);
            update_post_meta($post_id, '_manage_stock', 'yes');
        }
        if (!empty($data['image_url'])) {
           // Image logic stub
        }

        return $is_update ? 'updated' : 'created';
    }

    private function _import_single_order($data) {
        global $wpdb;
        $table = $wpdb->prefix . 'hng_orders';
        if (function_exists('hng_db_full_table_name')) $table = hng_db_full_table_name('hng_orders');

        // Mapping
        $order_number = $data['order_number'] ?? ($data['id'] ?? uniqid());
        $email = $data['email'] ?? '';
        $total = $data['total'] ?? 0;
        $status = $data['status'] ?? 'pending';
        $created_at = $data['created_at'] ?? current_time('mysql');
        $customer_name = $data['customer_name'] ?? '';

        if (empty($email) && empty($total)) return new WP_Error('invalid', 'Dados insuficientes');

        // Check exists
        $exists = $wpdb->get_var($wpdb->prepare("SELECT id FROM $table WHERE order_number = %s", $order_number));

        $db_data = [
            'order_number' => $order_number,
            'billing_email' => $email,
            'total' => $total,
            'status' => $status,
            'created_at' => $created_at,
            'billing_first_name' => $customer_name
        ];

        if ($exists) {
            $wpdb->update($table, $db_data, ['id' => $exists]);
            return 'updated';
        } else {
            $wpdb->insert($table, $db_data);
            return 'created';
        }
    }

    /**
     * Stub for batch processor if needed by cron (old ref)
     */
    public function process_import_batch($attachment_id) {
         // Placeholder to prevent fatal error if cron fires
    }

    /**
     * Exportar produtos como CSV (download)
     */
    public function export_products() {
        // Nonce via GET/REQUEST
        $nonce = isset($_REQUEST['nonce']) ? sanitize_text_field(wp_unslash($_REQUEST['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'hng-commerce-admin')) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- wp_die() handles escaping
            wp_die(__('Nonce inválido', 'hng-commerce'), '', 403);
        }

        if (!current_user_can('manage_options')) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- wp_die() handles escaping
            wp_die(__('Sem permissão', 'hng-commerce'), '', 403);
        }

        $filename = 'hng-products-' . gmdate('YmdHis') . '.csv';
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename=' . $filename);
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- UTF-8 BOM required for CSV encoding
        echo "\xEF\xBB\xBF"; // BOM

        $out = new SplTempFileObject();
        $out->fputcsv(['ID', 'Name', 'SKU', 'Price', 'Regular Price', 'Sale Price', 'Stock Quantity', 'Stock Status', 'Categories', 'Tags', 'Permalink', 'Image']);

        $posts = get_posts(['post_type' => 'hng_product', 'posts_per_page' => -1, 'post_status' => 'any']);
        foreach ($posts as $p) {
            $product = hng_get_product($p->ID);
            $cats = wp_get_post_terms($p->ID, 'hng_product_cat', ['fields' => 'names']);
            $tags = wp_get_post_terms($p->ID, 'hng_product_tag', ['fields' => 'names']);
            $row = [
                $product->get_id(),
                $product->get_name(),
                $product->get_sku(),
                $product->get_price(),
                $product->get_regular_price(),
                $product->get_sale_price(),
                $product->get_stock_quantity(),
                $product->get_stock_status(),
                implode(',', $cats),
                implode(',', $tags),
                $product->get_permalink(),
                $product->get_image_url('full')
            ];
            $out->fputcsv($row);
        }

        $out->rewind();
        while (!$out->eof()) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- CSV data output
            echo $out->fgets();
        }
        exit;
    }

    /**
     * Exportar pedidos como CSV (download)
     */
    public function export_orders() {
        $nonce = isset($_REQUEST['nonce']) ? sanitize_text_field(wp_unslash($_REQUEST['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'hng-commerce-admin')) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- wp_die() handles escaping
            wp_die(__('Nonce inválido', 'hng-commerce'), '', 403);
        }

        if (!current_user_can('manage_options')) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- wp_die() handles escaping
            wp_die(__('Sem permissão', 'hng-commerce'), '', 403);
        }

        global $wpdb;
            require_once HNG_COMMERCE_PATH . 'includes/helpers/hng-db.php';
            $safe_table = hng_db_full_table_name( 'hng_orders' );
            $table_sql = '`' . str_replace('`','', $safe_table) . '`';
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name sanitized via hng_db_full_table_name() and backtick escaping
            $orders = $wpdb->get_results( "SELECT * FROM {$table_sql} ORDER BY created_at DESC", ARRAY_A );

        $filename = 'hng-orders-' . gmdate('YmdHis') . '.csv';
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename=' . $filename);
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- UTF-8 BOM required for CSV encoding
        echo "\xEF\xBB\xBF"; // BOM

        $out = new SplTempFileObject();
        $out->fputcsv(['Order ID', 'Order Number', 'Customer ID', 'Email', 'Total', 'Status', 'Created At']);

        foreach ($orders as $o) {
            $email = isset($o['billing_email']) ? $o['billing_email'] : '';
            $row = [
                $o['id'],
                $o['order_number'] ?? '',
                $o['customer_id'] ?? '',
                $email,
                $o['total'] ?? '',
                $o['status'] ?? '',
                $o['created_at'] ?? ''
            ];
            $out->fputcsv($row);
        }

        $out->rewind();
        while (!$out->eof()) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- CSV data output
            echo $out->fgets();
        }
        exit;
    }
    
}